<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Rest_Base_Model extends App_Model
{
    protected string $tableName = '';
    protected string $primaryField = 'title';

    public function __construct(string $table_name)
    {
        $this->tableName = $table_name;
        parent::__construct();
    }

    /**
     * Get all
     * @param array $where
     * @param string $type
     * @return array
     */
    public function all(array $where = [], string $type = 'array',array $orderBy = []): array
    {
        if (count($where)) {
            $this->db->where($where);
        }
        if(count($orderBy)){
            foreach ($orderBy as $key => $value) {
                $this->db->order_by($value['order_by_field'], $value['order_by']);
            }
        }
        return $type === 'array' ? $this->db->get($this->tableName)->result_array() : $this->db->get($this->tableName)->result_object();
    }

    /**
     * Get by id
     * @param int $id
     * @param array $where
     * @param $select
     * @return object
     */
    public function get(int $id, array $where = [], $select = null): object
    {
        $this->db->select(is_null($select) ? '*' : $select);
        $this->db->where('id', $id);
        //$this->db->where($where);

        return $this->db->get($this->tableName)->row();
    }

    /**
     * Get key value array
     * @return array
     */
    public function getKeyValueArray($where=[]): array
    {
        $arr = [];
        $this->db->select(['id', $this->primaryField]);
        if(!empty($where)){
            $this->db->where($where);
        }
        $this->db->order_by($this->primaryField);
        foreach ($this->db->get($this->tableName)->result_array() as $row) {
            $arr[] = ['key' => $row['id'], 'value' => $row[$this->primaryField]];
        }

        return $arr;
    }

    /**
     * Add Data
     * @param array $data
     * @return int Inserted ID
     */
    public function add(array $data): int
    {
        $data['created_at'] = date('Y-m-d H:i:s');
        $data['updated_at'] = date('Y-m-d H:i:s');
        $this->db->insert($this->tableName, $data);

        return $this->db->insert_id();
    }

    /**
     * Update data
     * @param int $id
     * @param array $data
     * @return bool TRUE on success, FALSE on failure
     */
    public function update(int $id, array $data): bool
    {
        $data['updated_at'] = date('Y-m-d H:i:s');
        $this->db->where('id', $id);
        return $this->db->update($this->tableName, $data);
    }

    /**
     * @param array $arrData
     * @param array $uniqueBy
     * @param bool $timestamp
     * @return int
     */
    public function upsert(array $arrData, array $uniqueBy, bool $timestamp = false): int
    {
        if (empty($arrData) || empty($uniqueBy)) {
            return -1;
        }

        $arrFields = array_keys($arrData[0]);
        if ($timestamp) {
            $arrFields[] = 'created_at';
            $arrFields[] = 'updated_at';
            //$arrUpdate[] = 'updated_at';
        }

        $strKeys = implode(',', $arrFields);

        $arrDuplicateUpdate = [];
        foreach ($arrFields as $key) {
            if (!in_array($key, $uniqueBy) && $key !== 'created_at') {
                $arrDuplicateUpdate[] = $key . '=VALUES(' . $key . ')';
            }
        }

        $strDuplicateUpdate = implode(",", $arrDuplicateUpdate);

        $affected_rows = 0;
        $arrVal = [];
        foreach ($arrData as $data) {

            if ($timestamp) {
                $data['created_at'] = date('Y-m-d H:i:s');
                $data['updated_at'] = date('Y-m-d H:i:s');
            }
            $tempValues = array_map('addslashes', array_values($data));

            $arrValTemp = "'" . implode("','", $tempValues) . "'";
            $arrVal[] = str_replace("'NULL'", 'NULL', $arrValTemp);

            if (count($arrVal) > 1000) {

                $sql = "INSERT INTO " . $this->tableName . "  (" . $strKeys . ") " .
                    " VALUES(" . implode("),(", $arrVal) . ") " .
                    " ON DUPLICATE KEY UPDATE " . $strDuplicateUpdate;

                $this->db->query($sql);
                $arrVal = [];

                $affected_rows += $this->db->affected_rows();
            }
        }

        if (count($arrVal) > 0) {
            $sql = "INSERT INTO " . $this->tableName . "  (" . $strKeys . ") " .
                " VALUES(" . implode("),(", $arrVal) . ") " .
                " ON DUPLICATE KEY UPDATE " . $strDuplicateUpdate;

            $this->db->query($sql);
            $affected_rows += $this->db->affected_rows();
        }

        return $affected_rows;
    }

    /**
     * @param $ids
     * @return int
     */
    public function delete($ids): int
    {
        if (!is_array($ids)) {
            $ids = [$ids];
        }

        $this->db->where_in('id', $ids);
        $this->db->delete($this->tableName);

        return $this->db->affected_rows();
    }

    /**
     * Error Logging Interface
     *
     * We use this as a simple mechanism to access the logging
     * class and send messages to be logged.
     *
     * @param string    the error level: 'error', 'debug' or 'info'
     * @param string    the error message
     * @return    void
     */
    protected function log_message($level, $message): void
    {
        log_message($level, get_class($this) . '::' . debug_backtrace(!DEBUG_BACKTRACE_PROVIDE_OBJECT | DEBUG_BACKTRACE_IGNORE_ARGS, 2)[1]['function'] . ' ~> ' . $message);
    }
}
