<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Deals_model extends App_Model
{
    public function get($id = '')
    {
        if (!$id && staff_cant('view', 'deals')) {
            $staffid = get_staff_user_id();
            $this->db->where('(' . db_prefix() . 'deals.assigned = ' . $staffid . ' OR ' . db_prefix() . 'deals.addedfrom=' . $staffid . ' OR ' . db_prefix() . 'deals.is_public=1' . $this->_get_visibility_where($staffid) . ')');
        }
        if ($id) {
            $this->db->where('id', $id);
            return $this->db->get(db_prefix() . 'deals')->row_array();
        }

        $this->db->select(db_prefix() . 'deals.*, ' . db_prefix() . 'deals_pipelines.name as pipeline_name, ' . db_prefix() . 'deals_stages.name as stage_name, ' . db_prefix() . 'staff.firstname, ' . db_prefix() . 'staff.lastname');
        $this->db->select('(SELECT GROUP_CONCAT(' . db_prefix() . 'deals_pipelines.name) FROM ' . db_prefix() . 'deals_pipeline_relations 
            JOIN ' . db_prefix() . 'deals_pipelines ON ' . db_prefix() . 'deals_pipelines.id = ' . db_prefix() . 'deals_pipeline_relations.pipeline_id 
            WHERE ' . db_prefix() . 'deals_pipeline_relations.deal_id = ' . db_prefix() . 'deals.id) as pipeline_names');
        $this->db->select('(SELECT GROUP_CONCAT(' . db_prefix() . 'deals_stages.name) FROM ' . db_prefix() . 'deals_pipeline_relations 
            JOIN ' . db_prefix() . 'deals_stages ON ' . db_prefix() . 'deals_stages.id = ' . db_prefix() . 'deals_pipeline_relations.stage_id 
            WHERE ' . db_prefix() . 'deals_pipeline_relations.deal_id = ' . db_prefix() . 'deals.id) as stage_names');
        $this->db->select('(SELECT GROUP_CONCAT(' . db_prefix() . 'deals_stage_statuses.name) FROM ' . db_prefix() . 'deals_pipeline_relations 
            JOIN ' . db_prefix() . 'deals_stage_statuses ON ' . db_prefix() . 'deals_stage_statuses.id = ' . db_prefix() . 'deals_pipeline_relations.stage_status_id 
            WHERE ' . db_prefix() . 'deals_pipeline_relations.deal_id = ' . db_prefix() . 'deals.id) as stage_status_names');
        $this->db->select('(SELECT GROUP_CONCAT(CONCAT(' . db_prefix() . 'deals_pipelines.name, " - ", ' . db_prefix() . 'deals_stages.name, " - ", 
            IF(' . db_prefix() . 'deals_stage_statuses.name IS NOT NULL, CONCAT(' . db_prefix() . 'deals_stage_statuses.name, IF(' . db_prefix() . 'deals_pipeline_relations.score IS NOT NULL, CONCAT(" (", ' . db_prefix() . 'deals_pipeline_relations.score, ")"), "")), ""))) 
            FROM ' . db_prefix() . 'deals_pipeline_relations 
            JOIN ' . db_prefix() . 'deals_pipelines ON ' . db_prefix() . 'deals_pipelines.id = ' . db_prefix() . 'deals_pipeline_relations.pipeline_id 
            JOIN ' . db_prefix() . 'deals_stages ON ' . db_prefix() . 'deals_stages.id = ' . db_prefix() . 'deals_pipeline_relations.stage_id 
            LEFT JOIN ' . db_prefix() . 'deals_stage_statuses ON ' . db_prefix() . 'deals_stage_statuses.id = ' . db_prefix() . 'deals_pipeline_relations.stage_status_id 
            WHERE ' . db_prefix() . 'deals_pipeline_relations.deal_id = ' . db_prefix() . 'deals.id) as pipeline_relations');
        $this->db->join(db_prefix() . 'deals_pipelines', db_prefix() . 'deals.pipeline_id = ' . db_prefix() . 'deals_pipelines.id', 'left');
        $this->db->join(db_prefix() . 'deals_stages', db_prefix() . 'deals.stage_id = ' . db_prefix() . 'deals_stages.id', 'left');
        $this->db->join(db_prefix() . 'staff', db_prefix() . 'deals.assigned = ' . db_prefix() . 'staff.staffid', 'left');
        return $this->db->get(db_prefix() . 'deals')->result_array();
    }
    private function _get_visibility_where($staffid)
    {
        return " OR EXISTS (SELECT 1 FROM " . db_prefix() . "team_groups tg WHERE JSON_CONTAINS(" . db_prefix() . "deals.visibility_team_groups, CONCAT('\"', tg.id, '\"')) AND (JSON_CONTAINS(tg.manager_ids, '\"$staffid\"') OR JSON_CONTAINS(tg.tl_ids, '\"$staffid\"'))) 
                 OR EXISTS (SELECT 1 FROM " . db_prefix() . "team_sub_groups tsg WHERE JSON_CONTAINS(" . db_prefix() . "deals.visibility_sub_groups, CONCAT('\"', tsg.id, '\"')) AND (JSON_CONTAINS(tsg.tl_ids, '\"$staffid\"') OR JSON_CONTAINS(tsg.employee_ids, '\"$staffid\"')))";
    }
    public function update_visibility($id, $data)
    {
        $update = [
            'visibility_team_groups' => !empty($data['team_groups']) ? json_encode($data['team_groups']) : null,
            'visibility_sub_groups' => !empty($data['sub_groups']) ? json_encode($data['sub_groups']) : null,
        ];
        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'deals', $update);
        return $this->db->affected_rows() > 0;
    }

    // Update staff_can_access_deal($id)
    public function staff_can_access_deal($id)
    {
        if (staff_can('view', 'deals')) {
            return true;
        }

        $staffid = get_staff_user_id();
        $deal = $this->get($id);

        if (!$deal) {
            return false;
        }

        // Existing check
        if ($deal['assigned'] == $staffid || $deal['addedfrom'] == $staffid || $deal['is_public'] == 1) {
            return true;
        }
        // New visibility check
        if (!empty($deal['visibility_team_groups']) || !empty($deal['visibility_sub_groups'])) {
            $team_groups = json_decode($deal['visibility_team_groups'], true) ?? [];
            $sub_groups = json_decode($deal['visibility_sub_groups'], true) ?? [];
            // Debug: Check team groups
            if (!empty($team_groups)) {
                $this->db->where_in('id', $team_groups);
                $this->db->where('(JSON_CONTAINS(manager_ids, \'"' . $staffid . '"\') OR JSON_CONTAINS(tl_ids, \'"' . $staffid . '"\'))');
                $query = $this->db->get(db_prefix() . 'team_groups');
                // Debug output
                log_message('debug', 'Deals_model::staff_can_access_deal - Team groups query: ' . $this->db->last_query());
                log_message('debug', 'Deals_model::staff_can_access_deal - Team groups result count: ' . $query->num_rows());
                if ($query->num_rows() > 0) {
                    return true;
                }
            }

            // Debug: Check sub groups
            if (!empty($sub_groups)) {
                $this->db->where_in('id', $sub_groups);
                $this->db->where("(JSON_CONTAINS(tl_ids, '\"$staffid\"') OR JSON_CONTAINS(employee_ids, '\"$staffid\"'))");
                $query = $this->db->get(db_prefix() . 'team_sub_groups');
                // Debug output
                log_message('debug', 'Deals_model::staff_can_access_deal - Sub groups query: ' . $this->db->last_query());
                log_message('debug', 'Deals_model::staff_can_access_deal - Sub groups result count: ' . $query->num_rows());
                if ($query->num_rows() > 0) {
                    return true;
                }
            }
        }

        return false;
    }

    public function get_all_team_groups() {
        return $this->db->get(db_prefix() . 'team_groups')->result_array();
    }
    
    public function get_subgroups_by_groups($group_ids) {
        if (!is_array($group_ids)) {
            $group_ids = json_decode($group_ids, true) ?: [];
        }
        $this->db->where_in('group_id', $group_ids);
        $subgroups = $this->db->get(db_prefix() . 'team_sub_groups')->result_array();
        // Add selected flag based on deal's current subgroups
        $deal_subgroups = $this->db->where('deal_id', $this->input->post('deal_id'))
                                  ->get('tbldeal_subgroups')
                                  ->result_array();
        $selected_ids = array_column($deal_subgroups, 'subgroup_id');
        foreach ($subgroups as &$sg) {
            $sg['selected'] = in_array($sg['id'], $selected_ids);
        }
        return ['success' => true, 'subgroups' => $subgroups];
    }
    
    public function save_visibility($deal_id, $team_groups, $sub_groups) {
        $this->db->trans_start();
        // Clear existing visibility
        $this->db->where('deal_id', $deal_id)->delete('tbldeal_teamgroups');
        $this->db->where('deal_id', $deal_id)->delete('tbldeal_subgroups');
        // Save team groups
        if ($team_groups && is_array($team_groups)) {
            foreach ($team_groups as $group_id) {
                $this->db->insert('tbldeal_teamgroups', [
                    'deal_id' => $deal_id,
                    'group_id' => $group_id
                ]);
            }
        }
        // Save subgroups
        if ($sub_groups && is_array($sub_groups)) {
            foreach ($sub_groups as $subgroup_id) {
                $this->db->insert('tbldeal_subgroups', [
                    'deal_id' => $deal_id,
                    'subgroup_id' => $subgroup_id
                ]);
            }
        }
        $this->db->trans_complete();
        return ['success' => $this->db->trans_status()];
    }

    public function create_stage_status($data)
    {
        $this->db->insert(db_prefix() . 'deals_stage_statuses', $data);
        return $this->db->insert_id();
    }
    public function update_stage_status($id, $data)
    {
        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'deals_stage_statuses', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_stage_status($id)
    {
        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'deals_stage_statuses');
        return $this->db->affected_rows() > 0;
    }
    public function reorder_stage_statuses($stage_id, $order)
    {
        foreach ($order as $index => $status_id) {
            $this->db->where('id', $status_id);
            $this->db->where('stage_id', $stage_id);
            $this->db->update(db_prefix() . 'deals_stage_statuses', ['order' => $index + 1]);
        }
    }

    public function get_ratings()
    {
        return $this->db->order_by('score', 'asc')->get(db_prefix() . 'deals_ratings')->result_array();
    }
    public function get_stage_statuses($stage_id)
    {
        $this->db->where('stage_id', $stage_id);
        $this->db->order_by('order', 'asc');
        return $this->db->get(db_prefix() . 'deals_stage_statuses')->result_array();
    }

    public function get_stage_status($id)
    {
        return $this->db->where('id', (int) $id)->get(db_prefix() . 'deals_stage_statuses')->row_array();
    }

    public function get_column_order()
    {
        $columns = [
            ['id' => 'checkbox', 'name' => _l('select_all')],
            ['id' => 'deal-number', 'name' => _l('the_number_sign')],
            ['id' => 'deal-name', 'name' => _l('deal_name')],
            ['id' => 'deal-company', 'name' => _l('company')],
            ['id' => 'deal-pipeline', 'name' => _l('pipeline')],
            ['id' => 'deal-stage', 'name' => _l('stage')],
            ['id' => 'deal-all-pipelines', 'name' => _l('pipelines') . ' / ' . _l('stage') . ' / ' . _l('score')],
            ['id' => 'deal-value', 'name' => _l('deal_value')],
            ['id' => 'deal-assigned', 'name' => _l('assigned_to')],
            ['id' => 'deal-status', 'name' => _l('status')],
            ['id' => 'deal-created', 'name' => _l('dateadded')],
        ];

        // Temporarily disable per-staff saved column order. Returning the default
        // column order prevents client-side code from reapplying an old layout
        // (which previously caused DataTables to go blank after load).
        // To re-enable, restore the DB lookup below.

        $column_order = array_column($columns, 'id');

        // Direct JSON output
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode([
            'success' => true,
            'columns' => $columns,
            'column_order' => $column_order
        ]);
        exit;
    }
    public function save_column_order()
    {
        // Accepts column_order as a JSON array string or as an array
        $column_order = $this->input->post('column_order');

        // If it's a string, try to decode it
        if (is_string($column_order)) {
            $decoded = json_decode($column_order, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                $column_order = $decoded;
            }
        }

        // Validate: must be a non-empty array
        if (!is_array($column_order) || empty($column_order)) {
            return [
                'success' => false,
                'message' => _l('deals_column_order_update_failed')
            ];
        }

        $staff_id = get_staff_user_id();
        $data = [
            'staff_id' => $staff_id,
            'column_order' => json_encode($column_order),
            'dateupdated' => date('Y-m-d H:i:s')
        ];

        $this->db->where('staff_id', $staff_id);
        $existing = $this->db->get(db_prefix() . 'deals_column_order')->row();

        if ($existing) {
            $this->db->where('staff_id', $staff_id);
            $this->db->update(db_prefix() . 'deals_column_order', $data);
            $affected = $this->db->affected_rows();
        } else {
            $this->db->insert(db_prefix() . 'deals_column_order', $data);
            $affected = $this->db->affected_rows();
        }

        header('Content-Type: application/json; charset=utf-8');
        echo json_encode([
            'success' => $affected > 0,
            'message' => $affected > 0 ? _l('deals_column_order_updated') : _l('deals_column_order_update_failed')
        ]);
        exit;
    }
    /**
     * Get deals with applied filters
     * @param array $filters Filter parameters
     * @return array
     */
    public function get_deals($filters = [])
    {
        $this->db->select(db_prefix() . 'deals.*, ' . db_prefix() . 'deals_pipelines.name as pipeline_name, ' . db_prefix() . 'deals_stages.name as stage_name, ' . db_prefix() . 'deals_statuses.name as status_name, ' . get_sql_select_staff_full_names('assigned') . ' as assigned_fullname');
        $this->db->from(db_prefix() . 'deals');
        $this->db->join(db_prefix() . 'deals_pipelines', db_prefix() . 'deals.pipeline_id = ' . db_prefix() . 'deals_pipelines.id', 'left');
        $this->db->join(db_prefix() . 'deals_stages', db_prefix() . 'deals.stage_id = ' . db_prefix() . 'deals_stages.id', 'left');
        $this->db->join(db_prefix() . 'deals_statuses', db_prefix() . 'deals.status = ' . db_prefix() . 'deals_statuses.id', 'left');
        $this->db->join(db_prefix() . 'staff', db_prefix() . 'deals.assigned = ' . db_prefix() . 'staff.staffid', 'left');

        // Apply filters
        if (!empty($filters)) {
            foreach ($filters as $field => $value) {
                if (!empty($value)) {
                    switch ($field) {
                        case 'status_final':
                            $this->db->where_in(db_prefix() . 'deals.status_final', $value);
                            break;
                        case 'pipeline_id':
                            $this->db->where_in(db_prefix() . 'deals.pipeline_id', $value);
                            break;
                        case 'stage_id':
                            $this->db->where_in(db_prefix() . 'deals.stage_id', $value);
                            break;
                        case 'assigned':
                            $this->db->where_in(db_prefix() . 'deals.assigned', $value);
                            break;
                        case 'tags':
                            $this->db->where('FIND_IN_SET(' . $this->db->escape($value) . ',' . db_prefix() . 'deals.tags)');
                            break;
                        case 'dateadded':
                            $this->db->where('DATE(' . db_prefix() . 'deals.dateadded) >=', $value);
                            break;
                        case 'lastcontact':
                            $this->db->where('DATE(' . db_prefix() . 'deals.lastcontact) >=', $value);
                            break;
                        case 'deal_value':
                            $this->db->where(db_prefix() . 'deals.deal_value >=', $value);
                            break;
                    }
                }
            }
        }

        // Apply permissions & visibility for non-admin/non-view-all
        if (staff_cant('view', 'deals')) {
            $staffid = get_staff_user_id();
            $this->db->where('(' . db_prefix() . 'deals.assigned = ' . (int) $staffid
                . ' OR ' . db_prefix() . 'deals.addedfrom=' . (int) $staffid
                . ' OR ' . db_prefix() . 'deals.is_public=1' . $this->_get_visibility_where($staffid) . ')');
        }

        $this->db->order_by(db_prefix() . 'deals.id', 'DESC');
        return $this->db->get()->result_array();
    }

    /**
     * Get summary of deals by status
     * @return array
     */
    public function get_deals_summary()
    {
        $this->db->select('status_final, COUNT(*) as total, color');
        $this->db->from(db_prefix() . 'deals');
        $this->db->join(db_prefix() . 'deals_statuses', db_prefix() . 'deals.status = ' . db_prefix() . 'deals_statuses.id', 'left');
        $this->db->group_by('status_final');
        $statuses = $this->db->get()->result_array();

        $summary = [];
        foreach ($statuses as $status) {
            $summary[] = [
                'name' => $status['status_final'],
                'total' => $status['total'],
                'color' => $status['color'] ?? '#28B8DA',
                'percent' => round(($status['total'] / max(array_sum(array_column($statuses, 'total')), 1)) * 100, 2)
            ];
        }

        return $summary;
    }
    public function get_all($filters = [])
    {
        // Apply permissions & visibility for non-admin/non-view-all
        if (staff_cant('view', 'deals')) {
            $staffid = (string) get_staff_user_id();
            $team_group_ids = $this->db
                ->select('id')
                ->where("JSON_CONTAINS(manager_ids, '\"$staffid\"')", null, false)
                ->or_where("JSON_CONTAINS(tl_ids, '\"$staffid\"')", null, false)
                ->get(db_prefix() . 'team_groups')
                ->result_array();
            $team_group_ids = array_column($team_group_ids, 'id');

            $sub_group_ids = $this->db
                ->select('id')
                ->where("JSON_CONTAINS(tl_ids, '\"$staffid\"')", null, false)
                ->or_where("JSON_CONTAINS(employee_ids, '\"$staffid\"')", null, false)
                ->get(db_prefix() . 'team_sub_groups')
                ->result_array();
            $sub_group_ids = array_column($sub_group_ids, 'id');

            $visibility_sql = [];
            if (!empty($team_group_ids)) {
                $team_group_ids_escaped = array_map('strval', $team_group_ids);
                $visibility_sql[] = '(' . implode(' OR ', array_map(function ($id) {
                    return "JSON_CONTAINS(" . db_prefix() . "deals.visibility_team_groups, '\"$id\"')";
                }, $team_group_ids_escaped)) . ')';
            }
            if (!empty($sub_group_ids)) {
                $sub_group_ids_escaped = array_map('strval', $sub_group_ids);
                $visibility_sql[] = '(' . implode(' OR ', array_map(function ($id) {
                    return "JSON_CONTAINS(" . db_prefix() . "deals.visibility_sub_groups, '\"$id\"')";
                }, $sub_group_ids_escaped)) . ')';
            }

            $where = '(' . db_prefix() . 'deals.assigned = ' . (int) $staffid
                . ' OR ' . db_prefix() . 'deals.addedfrom=' . (int) $staffid
                . ' OR ' . db_prefix() . 'deals.is_public=1';

            if (!empty($visibility_sql)) {
                $where .= ' OR ' . implode(' OR ', $visibility_sql);
            }

            $where .= ')';

            $this->db->where($where);
        }

        // Apply filters
        foreach ($filters as $filter) {
            $type = isset($filter['type']) ? $filter['type'] : '';
            $value = isset($filter['value']) ? $filter['value'] : '';

            if ($type && $value !== '') {
                switch ($type) {
                    case 'pipeline':
                        $this->db->where(db_prefix() . 'deals.pipeline_id', (int) $value);
                        break;
                    case 'stage':
                        $this->db->where(db_prefix() . 'deals.stage_id', (int) $value);
                        break;
                    case 'status':
                        $this->db->where(db_prefix() . 'deals.status_final', $this->db->escape_str($value));
                        break;
                    case 'business_category':
                        $this->db->where(db_prefix() . 'deals.business_category_id', (int) $value);
                        break;
                    case 'source':
                        $this->db->where(db_prefix() . 'deals.source_id', (int) $value);
                        break;
                    case 'rating':
                        $this->db->where(db_prefix() . 'deals.rating_id', (int) $value);
                        break;
                }
            }
        }

        $deals = $this->db->order_by('dateadded', 'DESC')->get(db_prefix() . 'deals')->result_array();

        $pipelines = $this->db->get(db_prefix() . 'deals_pipelines')->result_array();
        $stages = $this->db->get(db_prefix() . 'deals_stages')->result_array();
        $staff = $this->db->select('staffid, firstname, lastname')->get(db_prefix() . 'staff')->result_array();

        $pipelineMap = [];
        foreach ($pipelines as $pipeline) {
            $pipelineMap[$pipeline['id']] = $pipeline['name'];
        }
        $stageMap = [];
        foreach ($stages as $stage) {
            $stageMap[$stage['id']] = $stage['name'];
        }
        $staffMap = [];
        foreach ($staff as $member) {
            $staffMap[$member['staffid']] = $member['firstname'] . ' ' . $member['lastname'];
        }

        foreach ($deals as &$deal) {
            $pipeline_id = $deal['pipeline_id'];
            $stage_id = $deal['stage_id'];
            $assigned_id = isset($deal['assigned']) ? $deal['assigned'] : null;

            $deal['pipeline'] = [
                'id' => $pipeline_id,
                'label' => isset($pipelineMap[$pipeline_id]) ? $pipelineMap[$pipeline_id] : null,
            ];
            $deal['stage'] = [
                'id' => $stage_id,
                'label' => isset($stageMap[$stage_id]) ? $stageMap[$stage_id] : null,
            ];
            if ($assigned_id !== null) {
                $deal['assigned'] = [
                    'id' => $assigned_id,
                    'label' => isset($staffMap[$assigned_id]) ? $staffMap[$assigned_id] : null,
                ];
            }

            $deal['pipelines'] = $this->get_deal_pipelines($deal['id']);

            unset($deal['pipeline_id'], $deal['stage_id']);
        }
        unset($deal);

        return $deals;
    }

    /**
     * Get deals table data with filters and all deal data (for datatable and filter usage)
     */
    public function get_table_data($postData = [])
    {
        // Define filter rules
        $rules = [
            App_table_filter::new('name', 'TextRule')->label(_l('deals_dt_name')),
            App_table_filter::new('company', 'TextRule')->label(_l('company')),
            App_table_filter::new('deal_value', 'NumberRule')->label(_l('deal_value')),
            App_table_filter::new('dateadded', 'DateRule')->label(_l('date_created')),
            App_table_filter::new('status_final', 'MultiSelectRule')->label(_l('status'))->options([
                ['value' => 'open', 'label' => _l('open')],
                ['value' => 'won', 'label' => _l('won')],
                ['value' => 'lost', 'label' => _l('lost')],
            ]),
            App_table_filter::new('pipeline_id', 'SelectRule')->label(_l('pipeline'))->options(function () {
                $pipelines = $this->db->get(db_prefix() . 'deals_pipelines')->result_array();
                return collect($pipelines)->map(fn($p) => ['value' => $p['id'], 'label' => $p['name']])->all();
            }),
            App_table_filter::new('stage_id', 'SelectRule')->label(_l('stage'))->options(function () {
                $stages = $this->db->get(db_prefix() . 'deals_stages')->result_array();
                return collect($stages)->map(fn($s) => ['value' => $s['id'], 'label' => $s['name']])->all();
            }),
            App_table_filter::new('status', 'SelectRule')->label(_l('leads_dt_status'))->options(function () {
                $statuses = $this->get_statuses();
                return collect($statuses)->map(fn($s) => ['value' => $s['id'], 'label' => $s['name']])->all();
            }),
            App_table_filter::new('source', 'SelectRule')->label(_l('leads_source'))->options(function () {
                $sources = $this->get_sources();
                return collect($sources)->map(fn($s) => ['value' => $s['id'], 'label' => $s['name']])->all();
            }),
        ];

        $rules[] = App_table_filter::new('assigned', 'SelectRule')->label(_l('assigned_to'))
            ->withEmptyOperators()->emptyOperatorValue(0)
            ->isVisible(fn () => staff_can('view', 'deals'))
            ->options(function () {
                $staff = $this->staff_model->get('', ['active' => 1]);
                return collect($staff)->map(fn($m) => ['value' => $m['staffid'], 'label' => $m['firstname'] . ' ' . $m['lastname']])->all();
            });

        // Define table columns (order must match the table header in the view)
        $aColumns = [
            '1', // For checkbox placeholder
            db_prefix() . 'deals.id as id',
            db_prefix() . 'deals.name as name',
            'company',
            'dp.name as pipeline_name',
            'ds.name as stage_name',
            // All pipelines / stage / score for multi-pipeline relations
            '(SELECT GROUP_CONCAT(CONCAT(p2.name, " - ", s2.name, IF(r2.score IS NOT NULL, CONCAT(" (", r2.score, ")"), "")) SEPARATOR "; ") FROM ' . db_prefix() . 'deals_pipeline_relations r2 JOIN ' . db_prefix() . 'deals_pipelines p2 ON p2.id = r2.pipeline_id JOIN ' . db_prefix() . 'deals_stages s2 ON s2.id = r2.stage_id LEFT JOIN ' . db_prefix() . 'deals_stage_statuses ss2 ON ss2.id = r2.stage_status_id WHERE r2.deal_id = ' . db_prefix() . 'deals.id) as pipeline_relations',
            db_prefix() . 'deals.deal_value as deal_value',
            // Assigned staff full name
            'CONCAT(s.firstname, " ", s.lastname) as assigned',
            'status_final',
            'dateadded',
        ];

        $sIndexColumn = 'id';
        $sTable = db_prefix() . 'deals';
        $join = [
            'LEFT JOIN ' . db_prefix() . 'deals_pipelines dp ON dp.id = ' . db_prefix() . 'deals.pipeline_id',
            'LEFT JOIN ' . db_prefix() . 'deals_stages ds ON ds.id = ' . db_prefix() . 'deals.stage_id',
            'LEFT JOIN ' . db_prefix() . 'staff s ON s.staffid = ' . db_prefix() . 'deals.assigned',
        ];

        $where = [];
        // Apply filters from App_table
        $filtersWhere = App_table::find('deals')->getWhereFromRules($postData);
        if ($filtersWhere) {
            $where[] = $filtersWhere;
        }

        // Apply permission-based filtering
        if (staff_cant('view', 'deals')) {
            $staffid = get_staff_user_id();
            $where[] = 'AND (' . db_prefix() . 'deals.assigned = ' . (int) $staffid . ' OR ' . db_prefix() . 'deals.addedfrom=' . (int) $staffid . ' OR ' . db_prefix() . 'deals.is_public=1)';
        }

        $additionalColumns = [
            db_prefix() . 'deals.addedfrom as addedfrom',
        ];

        // Initialize DataTable
        $result = data_tables_init($aColumns, $sIndexColumn, $sTable, $join, $where, $additionalColumns);
        $output = $result['output'];
        $rResult = $result['rResult'];

        // Format the output for DataTables
        foreach ($rResult as $aRow) {
            $row = [];
            // Checkbox
            $row[] = '<div class="checkbox"><input type="checkbox" value="' . $aRow['id'] . '"><label></label></div>';
            // ID
            $hrefAttr = 'href="' . admin_url('deals/view/' . $aRow['id']) . '"';
            $row[] = '<a ' . $hrefAttr . ' class="tw-font-medium">' . $aRow['id'] . '</a>';
            // Name with row options
            $nameRow = '<a ' . $hrefAttr . ' class="tw-font-medium">' . e($aRow['name']) . '</a>';
            $nameRow .= '<div class="row-options">';
            $nameRow .= '<a ' . $hrefAttr . '>' . _l('view') . '</a>';
            $nameRow .= ' | <a href="' . admin_url('deals/edit/' . $aRow['id']) . '">' . _l('edit') . '</a>';
            $nameRow .= ' | <a href="' . admin_url('deals/delete/' . $aRow['id']) . '" class="_delete text-danger">' . _l('delete') . '</a>';
            $nameRow .= '</div>';
            $row[] = $nameRow;
            // Company
            $row[] = e($aRow['company']);
            // Note: pipeline_name and stage_name columns were removed from the view.
            // The combined pipelines/stage/score column is rendered below as HTML list.
            // All pipelines / stage / score - render as list with highlighted pipeline names
            if (!empty($aRow['pipeline_relations'])) {
                // pipeline_relations stored as '; ' separated list of "Pipeline - Stage (score)" entries
                $parts = explode('; ', $aRow['pipeline_relations']);
                $listHtml = '<ul class="deals-pipeline-relations-list" style="list-style:none;padding:0;margin:0;">';
                foreach ($parts as $p) {
                    $p = trim($p);
                    if ($p === '') continue;
                    // Expect format: "Pipeline - Stage (score)" or "Pipeline - Stage"
                    $segments = explode(' - ', $p, 3);
                    $pipelineName = isset($segments[0]) ? htmlspecialchars($segments[0], ENT_QUOTES, 'UTF-8') : '';
                    $stageAndScore = isset($segments[1]) ? htmlspecialchars($segments[1], ENT_QUOTES, 'UTF-8') : '';
                    if (isset($segments[2])) {
                        $stageAndScore .= ' - ' . htmlspecialchars($segments[2], ENT_QUOTES, 'UTF-8');
                    }
                    $listHtml .= '<li style="margin-bottom:4px;"><span class="label label-default deals-pipeline-name" style="background:#f0f6ff;color:#09326a;margin-right:6px;padding:3px 6px;border-radius:3px;">' . $pipelineName . '</span>' . $stageAndScore . '</li>';
                }
                $listHtml .= '</ul>';
                $row[] = $listHtml;
            } else {
                $row[] = '-';
            }
            // Deal value
            $row[] = '<span class="tw-font-medium">' . app_format_money((float) ($aRow['deal_value'] ?? 0), get_base_currency()->id) . '</span>';
            // Assigned
            $row[] = isset($aRow['assigned']) && $aRow['assigned'] !== ' ' ? e($aRow['assigned']) : '-';
            // Status
            $row[] = ucfirst($aRow['status_final']);
            // Date added
            $row[] = '<span data-toggle="tooltip" data-title="' . e(_dt($aRow['dateadded'])) . '" class="text-has-action is-date">' . e(time_ago($aRow['dateadded'])) . '</span>';
            $row['DT_RowId'] = 'deal_' . $aRow['id'];
            $output['aaData'][] = $row;
        }

        // Also send all deals data (for filters/datatable usage)
        // Convert postData filters to get_all format
        $filters = [];
        if (isset($postData['filters']) && is_array($postData['filters'])) {
            foreach ($postData['filters'] as $type => $value) {
                if ($value !== '' && $value !== null) {
                    $filters[] = ['type' => $type, 'value' => $value];
                }
            }
        }
        $output['all_deals'] = $this->get_all($filters);

        return $output;
    }
    public function update_deal_stage($deal_id, $pipeline_id, $stage_id)
    {
        $this->db->where('deal_id', $deal_id);
        $this->db->where('pipeline_id', $pipeline_id);
        $this->db->update(db_prefix() . 'deals_pipeline_relations', ['stage_id' => $stage_id]);
        return $this->db->affected_rows() > 0;
    }

    /**
     * Return all pipelines assigned to a deal with current stage and score
     */
    public function get_deal_pipelines($deal_id)
    {
        $this->db->select('r.id as relation_id, r.pipeline_id, r.stage_id, r.rating_id, r.score, dp.name as pipeline_name, ds.name as stage_name, dr.name as rating_name')
            ->from(db_prefix() . 'deals_pipeline_relations r')
            ->join(db_prefix() . 'deals_pipelines dp', 'dp.id = r.pipeline_id', 'left')
            ->join(db_prefix() . 'deals_stages ds', 'ds.id = r.stage_id', 'left')
            ->join(db_prefix() . 'deals_ratings dr', 'dr.id = r.rating_id', 'left')
            ->where('r.deal_id', (int) $deal_id)
            ->order_by('dp.`order`', 'ASC');

        $rows = $this->db->get()->result_array();
        return array_map(function ($row) {
            return [
                'relation_id' => (int) $row['relation_id'],
                'pipeline' => ['id' => (int) $row['pipeline_id'], 'label' => $row['pipeline_name']],
                'stage' => ['id' => (int) $row['stage_id'], 'label' => $row['stage_name']],
                'rating' => ['id' => (int) $row['rating_id'], 'label' => $row['rating_name']],
                'score' => $row['score'] !== null ? (int) $row['score'] : null,
            ];
        }, $rows);
    }

    /** Assign pipeline to deal (no duplicates) */
    public function add_pipeline_to_deal($deal_id, $pipeline_id, $stage_id = null, $rating_id = null, $score = null)
    {
        // Check if exists
        $exists = $this->db->where(['deal_id' => (int) $deal_id, 'pipeline_id' => (int) $pipeline_id])
            ->get(db_prefix() . 'deals_pipeline_relations')->row();
        if ($exists) {
            return (int) $exists->id;
        }
        $payload = [
            'deal_id' => (int) $deal_id,
            'pipeline_id' => (int) $pipeline_id,
            'stage_id' => $stage_id ? (int) $stage_id : null,
            'rating_id' => $rating_id ? (int) $rating_id : null,
            'score' => $score === null ? null : (int) $score,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id(),
        ];
        $this->db->insert(db_prefix() . 'deals_pipeline_relations', $payload);
        return $this->db->insert_id();
    }

    public function update_pipeline_relation($relation_id, $data)
    {
        $update = [];
        foreach (['stage_id', 'rating_id', 'score'] as $key) {
            if (array_key_exists($key, $data)) {
                $update[$key] = $data[$key] === null ? null : (int) $data[$key];
            }
        }
        if (empty($update)) {
            return true;
        }
        $this->db->where('id', (int) $relation_id)->update(db_prefix() . 'deals_pipeline_relations', $update);
        return $this->db->affected_rows() > 0;
    }

    public function remove_pipeline_relation($relation_id)
    {
        $this->db->where('id', (int) $relation_id)->delete(db_prefix() . 'deals_pipeline_relations');
        return $this->db->affected_rows() > 0;
    }

    public function get_stages_with_deals($pipelineId = null)
    {
        if (!$pipelineId) {
            $pipelineId = $this->get_default_pipeline_id();
        }
        $stages = $this->db->where('pipeline_id', (int) $pipelineId)->order_by('position', 'ASC')->get(db_prefix() . 'deals_stages')->result_array();
        foreach ($stages as &$stage) {
            $stage['deals'] = $this->db->where('stage_id', (int) $stage['id'])->order_by('dateadded', 'DESC')->get(db_prefix() . 'deals')->result_array();
        }
        return $stages;
    }

    public function get_one($id)
    {
        return $this->db->where('id', (int) $id)->get(db_prefix() . 'deals')->row_array();
    }

    public function update_deal($id, array $data)
    {
        // Remove any non-scalar fields (e.g. multi-pipelines array)
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                unset($data[$key]);
            }
        }
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals');
        return $this->db->affected_rows() > 0;
    }

    public function add_attachment($data)
    {
        $this->db->insert(db_prefix() . 'deals_attachments', $data);
        return $this->db->insert_id();
    }

    public function get_attachments($deal_id)
    {
        return $this->db->where('deal_id', (int) $deal_id)->get(db_prefix() . 'deals_attachments')->result_array();
    }



    public function create($data)
    {
        $insert = [
            'name' => $data['name'] ?? null,
            'company' => $data['company'] ?? null,
            'description' => $data['description'] ?? null,
            'pipeline_id' => $data['pipeline_id'] ?? $this->get_default_pipeline_id(),
            'stage_id' => $data['stage_id'] ?? $this->get_default_open_stage_id(),
            'deal_value' => $data['deal_value'] ?? 0,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id(),
            'status_final' => 'open',
            'source' => $data['source'] ?? null, // New field
            'status' => $data['status'] ?? null, // status ID
            'assigned' => $data['assigned'] ?? 0, // New field
        ];
        $this->db->insert(db_prefix() . 'deals', $insert);
        return $this->db->insert_id();
    }

    // public function update_status($id, $status)
    // {
    //     $statusId = $this->map_status_to_stage($status);
    //     $payload = [
    //         'last_status_change' => date('Y-m-d H:i:s'),
    //         'status' => $statusId,
    //     ];
    //     if ($status === 'won' || $status === 'lost') {
    //         $payload['status_final'] = $status;
    //     }
    //     $this->db->where('id', $id)->update(db_prefix() . 'deals', $payload);
    //     return $this->db->affected_rows() > 0;
    // }


    public function mark_lost($id, $reasonId = null)
    {
        $this->db->where('id', $id)->update(db_prefix() . 'deals', [
            'status_final' => 'lost',
            'lost_reason_id' => $reasonId,
            'last_status_change' => date('Y-m-d H:i:s'),
        ]);
        return $this->db->affected_rows() > 0;
    }
    public function mark_won($id)
    {
        $this->db->where('id', $id)->update(db_prefix() . 'deals', [
            'status_final' => 'won',
            'last_status_change' => date('Y-m-d H:i:s'),
        ]);
        return $this->db->affected_rows() > 0;
    }

    private function map_status_to_stage($status)
    {
        $statusName = null;
        if ($status === 'won') {
            $statusName = 'Won';
        } elseif ($status === 'lost') {
            $statusName = 'Lost';
        } else {
            $statusName = 'Open';
        }

        $row = $this->db->where('name', $statusName)->order_by('position', 'ASC')->limit(1)->get(db_prefix() . 'deals_stages')->row();
        return $row ? (int) $row->id : $this->get_default_open_stage_id();
    }

    // Pipelines helpers
    public function get_default_pipeline_id()
    {
        $row = $this->db->order_by('`order`', 'ASC')->limit(1)->get(db_prefix() . 'deals_pipelines')->row();
        return $row ? (int) $row->id : null;
    }

    public function get_default_open_stage_id()
    {
        $pipelineId = $this->get_default_pipeline_id();
        if (!$pipelineId) {
            return null;
        }
        $row = $this->db->where(['pipeline_id' => $pipelineId, 'is_won' => 0, 'is_lost' => 0])->order_by('position', 'ASC')->limit(1)->get(db_prefix() . 'deals_stages')->row();
        return $row ? (int) $row->id : null;
    }

    public function get_pipelines()
    {
        return $this->db->order_by('`order`', 'ASC')->get(db_prefix() . 'deals_pipelines')->result_array();
    }

    public function get_pipeline($id)
    {
        return $this->db->where('id', (int) $id)->get(db_prefix() . 'deals_pipelines')->row_array();
    }

    public function create_pipeline($name)
    {
        $this->db->insert(db_prefix() . 'deals_pipelines', ['name' => $name, 'order' => (int) ($this->db->count_all(db_prefix() . 'deals_pipelines') + 1)]);
        return $this->db->insert_id();
    }

    public function update_pipeline($id, $data)
    {
        $this->db->where('id', $id)->update(db_prefix() . 'deals_pipelines', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_pipeline($id)
    {
        $this->db->where('id', $id)->delete(db_prefix() . 'deals_pipelines');
        return $this->db->affected_rows() > 0;
    }

    public function get_stages($pipelineId)
    {
        if (staff_cant('view', 'deals')) {
            $staffid = get_staff_user_id();
            $where[] = 'AND (' . db_prefix() . 'deals.assigned = ' . $staffid . ' OR ' . db_prefix() . 'deals.addedfrom=' . $staffid . ' OR ' . db_prefix() . 'deals.is_public=1' . $this->_get_visibility_where($staffid) . ')';
        }
        return $this->db->where('pipeline_id', (int) $pipelineId)->order_by('position', 'ASC')->get(db_prefix() . 'deals_stages')->result_array();
    }

    public function create_stage($pipelineId, $data)
    {
        $data['pipeline_id'] = (int) $pipelineId;
        if (!isset($data['position'])) {
            $data['position'] = (int) $this->db->where('pipeline_id', (int) $pipelineId)->count_all_results(db_prefix() . 'deals_stages') + 1;
        }
        $this->db->insert(db_prefix() . 'deals_stages', $data);
        return $this->db->insert_id();
    }

    public function update_stage($id, $data)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals_stages', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_stage($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals_stages');
        return $this->db->affected_rows() > 0;
    }

    public function reorder_stages($pipelineId, $orderedIds)
    {
        $position = 1;
        foreach ((array) $orderedIds as $stageId) {
            $this->db->where('id', (int) $stageId)->where('pipeline_id', (int) $pipelineId)->update(db_prefix() . 'deals_stages', ['position' => $position]);
            $position++;
        }
        return true;
    }

    // Lost reasons CRUD
    public function get_lost_reasons()
    {
        return $this->db->order_by('name', 'ASC')->get(db_prefix() . 'deals_lost_reasons')->result_array();
    }

    public function get_lost_reason($id)
    {
        return $this->db->where('id', (int) $id)->get(db_prefix() . 'deals_lost_reasons')->row_array();
    }

    public function get_staff()
    {
        return $this->db->select('staffid, firstname, lastname')
            ->where('active', 1)
            ->order_by('firstname', 'ASC')
            ->get(db_prefix() . 'staff')
            ->result_array();
    }



    /**
     * Get the stage_id for a deal in a specific pipeline.
     * If not found, fallback to the main deal stage_id.
     *
     * @param int $deal_id
     * @param int $pipeline_id
     * @return int|null
     */
    public function get_deal_pipeline_stage($deal_id, $pipeline_id)
    {
        $row = $this->db->where('deal_id', (int) $deal_id)
            ->where('pipeline_id', (int) $pipeline_id)
            ->get(db_prefix() . 'deals_pipeline_relations')
            ->row_array();
        if ($row && isset($row['stage_id'])) {
            return (int) $row['stage_id'];
        }
        // Fallback to main deal stage_id
        $deal = $this->db->where('id', (int) $deal_id)->get(db_prefix() . 'deals')->row_array();
        return $deal ? (int) $deal['stage_id'] : null;
    }

    /**
     * Update the stage_id for a deal in a specific pipeline.
     * If the relation does not exist, create it.
     *
     * @param int $deal_id
     * @param int $pipeline_id
     * @param int $stage_id
     * @return bool
     */
    public function update_deal_pipeline_stage($deal_id, $pipeline_id, $stage_id)
    {
        $exists = $this->db->where('deal_id', (int) $deal_id)
            ->where('pipeline_id', (int) $pipeline_id)
            ->get(db_prefix() . 'deals_pipeline_relations')
            ->row_array();
        if ($exists) {
            $this->db->where('deal_id', (int) $deal_id)
                ->where('pipeline_id', (int) $pipeline_id)
                ->update(db_prefix() . 'deals_pipeline_relations', ['stage_id' => (int) $stage_id]);
        } else {
            $this->db->insert(db_prefix() . 'deals_pipeline_relations', [
                'deal_id' => (int) $deal_id,
                'pipeline_id' => (int) $pipeline_id,
                'stage_id' => (int) $stage_id
            ]);
        }
        return true;
    }

    public function update_deal_pipeline_stage_with_meta($deal_id, $pipeline_id, $stage_id, $stage_status_id = null, $stage_comment = null)
    {
        $payload = [
            'stage_id' => (int) $stage_id,
            'stage_status_id' => $stage_status_id ? (int) $stage_status_id : null,
            'stage_comment' => $stage_comment !== null ? $stage_comment : null,
        ];
        $exists = $this->db->where('deal_id', (int) $deal_id)
            ->where('pipeline_id', (int) $pipeline_id)
            ->get(db_prefix() . 'deals_pipeline_relations')
            ->row_array();
        if ($exists) {
            $this->db->where('deal_id', (int) $deal_id)
                ->where('pipeline_id', (int) $pipeline_id)
                ->update(db_prefix() . 'deals_pipeline_relations', $payload);
        } else {
            $payload['deal_id'] = (int) $deal_id;
            $payload['pipeline_id'] = (int) $pipeline_id;
            $payload['dateadded'] = date('Y-m-d H:i:s');
            $payload['addedfrom'] = get_staff_user_id();
            $this->db->insert(db_prefix() . 'deals_pipeline_relations', $payload);
        }
        return true;
    }

    public function get_relation_for_pipeline($deal_id, $pipeline_id)
    {
        return $this->db->where('deal_id', (int) $deal_id)
            ->where('pipeline_id', (int) $pipeline_id)
            ->get(db_prefix() . 'deals_pipeline_relations')
            ->row_array();
    }

    public function add_stage_change_history($deal_id, $pipeline_id, $from_stage_id, $to_stage_id, $stage_status_id = null, $comment = null)
    {
        $this->db->insert(db_prefix() . 'deals_stage_changes', [
            'deal_id' => (int) $deal_id,
            'pipeline_id' => (int) $pipeline_id,
            'from_stage_id' => $from_stage_id ? (int) $from_stage_id : null,
            'to_stage_id' => (int) $to_stage_id,
            'stage_status_id' => $stage_status_id ? (int) $stage_status_id : null,
            'comment' => $comment,
            'staffid' => get_staff_user_id(),
            'dateadded' => date('Y-m-d H:i:s'),
        ]);
        return $this->db->insert_id();
    }

    public function get_stage_name($stage_id)
    {
        $stage = $this->db->where('id', (int) $stage_id)->get(db_prefix() . 'deals_stages')->row_array();
        return $stage ? $stage['name'] : 'Unknown Stage';
    }

    public function create_lost_reason($name)
    {
        $this->db->insert(db_prefix() . 'deals_lost_reasons', ['name' => $name]);
        return $this->db->insert_id();
    }
    public function update_lost_reason($id, $name)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals_lost_reasons', ['name' => $name]);
        return $this->db->affected_rows() > 0;
    }
    public function delete_lost_reason($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals_lost_reasons');
        return $this->db->affected_rows() > 0;
    }

    // Deal Contacts Methods
    public function get_deal_contacts($deal_id)
    {
        $this->db->select('cb.*, l.dateadded as linked_date')
            ->from(db_prefix() . 'deal_contact_links l')
            ->join(db_prefix() . 'deals_contacts_book cb', 'cb.id = l.contact_id', 'left')
            ->where('l.deal_id', (int) $deal_id)
            ->where('cb.active', 1)
            ->order_by('l.dateadded', 'DESC');
        return $this->db->get()->result_array();
    }

    public function get_contact($contact_id)
    {
        return $this->db->where('id', (int) $contact_id)
            ->get(db_prefix() . 'deals_contacts_book')
            ->row_array();
    }

    public function add_contact_to_deal($deal_id, $contact_id)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'contact_id' => (int) $contact_id,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id()
        ];
        $this->db->insert(db_prefix() . 'deal_contact_links', $data);
        return $this->db->insert_id();
    }

    public function remove_contact_from_deal($deal_id, $contact_id)
    {
        $this->db->where('deal_id', (int) $deal_id)
            ->where('contact_id', (int) $contact_id)
            ->delete(db_prefix() . 'deal_contact_links');
        return $this->db->affected_rows() > 0;
    }

    public function get_available_contacts($deal_id = null)
    {
        $this->db->select('cb.*')
            ->from(db_prefix() . 'deals_contacts_book cb')
            ->where('cb.active', 1);
        if ($deal_id) {
            $this->db->where('cb.id NOT IN (SELECT contact_id FROM ' . db_prefix() . 'deal_contact_links WHERE deal_id = ' . (int) $deal_id . ')');
        }
        return $this->db->order_by('cb.firstname', 'ASC')->get()->result_array();
    }

    public function create_module_contact($data)
    {
        $insert = [
            'firstname' => $data['firstname'] ?? '',
            'lastname' => $data['lastname'] ?? '',
            'email' => $data['email'] ?? '',
            'phonenumber' => $data['phonenumber'] ?? '',
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id(),
            'active' => 1,
        ];
        $this->db->insert(db_prefix() . 'deals_contacts_book', $insert);
        return $this->db->insert_id();
    }

    /**
     * Edit a contact in the deals_contacts_book table.
     * @param int $contact_id
     * @param array $data
     * @return bool
     */
    public function edit_module_contact($contact_id, $data)
    {
        $update = [];
        if (isset($data['firstname'])) {
            $update['firstname'] = $data['firstname'];
        }
        if (isset($data['lastname'])) {
            $update['lastname'] = $data['lastname'];
        }
        if (isset($data['email'])) {
            $update['email'] = $data['email'];
        }
        if (isset($data['phonenumber'])) {
            $update['phonenumber'] = $data['phonenumber'];
        }
        if (isset($data['active'])) {
            $update['active'] = (int)$data['active'];
        }
        if (empty($update)) {
            return false;
        }
        $this->db->where('id', (int)$contact_id)->update(db_prefix() . 'deals_contacts_book', $update);
        return $this->db->affected_rows() > 0;
    }

    // Deal Companies Methods
    public function get_deal_companies($deal_id)
    {
        $this->db->select('cb.*, l.dateadded as linked_date')
            ->from(db_prefix() . 'deal_company_links l')
            ->join(db_prefix() . 'deals_companies_book cb', 'cb.id = l.company_id', 'left')
            ->where('l.deal_id', (int) $deal_id)
            ->where('cb.active', 1)
            ->order_by('l.dateadded', 'DESC');
        return $this->db->get()->result_array();
    }

    public function add_company_to_deal($deal_id, $company_id)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'company_id' => (int) $company_id,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id()
        ];
        $this->db->insert(db_prefix() . 'deal_company_links', $data);
        return $this->db->insert_id();
    }

    public function remove_company_from_deal($deal_id, $company_id)
    {
        $this->db->where('deal_id', (int) $deal_id)
            ->where('company_id', (int) $company_id)
            ->delete(db_prefix() . 'deal_company_links');
        return $this->db->affected_rows() > 0;
    }

    public function get_available_companies($deal_id = null)
    {
        $this->db->select('cb.*')
            ->from(db_prefix() . 'deals_companies_book cb')
            ->where('cb.active', 1);
        if ($deal_id) {
            $this->db->where('cb.id NOT IN (SELECT company_id FROM ' . db_prefix() . 'deal_company_links WHERE deal_id = ' . (int) $deal_id . ')');
        }
        return $this->db->order_by('cb.name', 'ASC')->get()->result_array();
    }

    // Company CRUD Methods (module-specific)
    public function create_company($data)
    {
        // Check if email or phone already exists
        if (!empty($data['email'])) {
            $existing = $this->db->where('email', $data['email'])
                ->where('active', 1)
                ->get(db_prefix() . 'deals_companies_book')
                ->row_array();
            if ($existing) {
                return ['error' => 'Email already exists', 'existing_id' => $existing['id']];
            }
        }

        if (!empty($data['phone'])) {
            $existing = $this->db->where('phone', $data['phone'])
                ->where('active', 1)
                ->get(db_prefix() . 'deals_companies_book')
                ->row_array();
            if ($existing) {
                return ['error' => 'Phone number already exists', 'existing_id' => $existing['id']];
            }
        }

        $insert = [
            'name' => $data['name'] ?? '',
            'email' => $data['email'] ?? '',
            'phone' => $data['phone'] ?? '',
            'website' => $data['website'] ?? '',
            'address' => $data['address'] ?? '',
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id(),
            'active' => 1
        ];

        $this->db->insert(db_prefix() . 'deals_companies_book', $insert);
        if ($this->db->affected_rows() > 0) {
            return $this->db->insert_id();
        }
        return false;
    }

    public function get_company($id)
    {
        return $this->db->where('id', (int) $id)->get(db_prefix() . 'deals_companies_book')->row_array();
    }

    public function update_company($id, $data)
    {
        // Only update if there is something to update
        $update = [];
        if (isset($data['name'])) {
            $update['name'] = $data['name'];
        }
        if (isset($data['email'])) {
            // Check for duplicate email (excluding current company)
            $existing = $this->db->where('email', $data['email'])
                ->where('active', 1)
                ->where('id !=', (int)$id)
                ->get(db_prefix() . 'deals_companies_book')
                ->row_array();
            if ($existing) {
                return ['error' => 'Email already exists', 'existing_id' => $existing['id']];
            }
            $update['email'] = $data['email'];
        }
        if (isset($data['phone'])) {
            // Check for duplicate phone (excluding current company)
            $existing = $this->db->where('phone', $data['phone'])
                ->where('active', 1)
                ->where('id !=', (int)$id)
                ->get(db_prefix() . 'deals_companies_book')
                ->row_array();
            if ($existing) {
                return ['error' => 'Phone number already exists', 'existing_id' => $existing['id']];
            }
            $update['phone'] = $data['phone'];
        }
        if (isset($data['website'])) {
            $update['website'] = $data['website'];
        }
        if (isset($data['address'])) {
            $update['address'] = $data['address'];
        }
        if (isset($data['active'])) {
            $update['active'] = $data['active'];
        }

        if (empty($update)) {
            return false;
        }

        $this->db->where('id', (int)$id)->update(db_prefix() . 'deals_companies_book', $update);
        return $this->db->affected_rows() > 0;
    }

    public function delete_company($id)
    {
        $this->db->where('id', (int) $id)->delete(db_prefix() . 'deals_companies_book');
        return $this->db->affected_rows() > 0;
    }

    // Deal Attachments Relations Methods
    public function get_deal_attachments($deal_id)
    {
        $this->db->select('da.*, dar.dateadded as linked_date')
            ->from(db_prefix() . 'deal_attachments_relations dar')
            ->join(db_prefix() . 'deals_attachments da', 'da.id = dar.attachment_id', 'left')
            ->where('dar.deal_id', (int) $deal_id)
            ->order_by('dar.dateadded', 'DESC');
        return $this->db->get()->result_array();
    }

    public function get_attachment($attachment_id)
    {
        return $this->db->where('id', (int) $attachment_id)->get(db_prefix() . 'deals_attachments')->row_array();
    }

    public function add_attachment_to_deal($deal_id, $attachment_id)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'attachment_id' => (int) $attachment_id,
            'dateadded' => date('Y-m-d H:i:s'),
            'addedfrom' => get_staff_user_id()
        ];

        $this->db->insert(db_prefix() . 'deal_attachments_relations', $data);
        return $this->db->insert_id();
    }

    public function remove_attachment_from_deal($deal_id, $attachment_id)
    {
        $this->db->where('deal_id', (int) $deal_id)
            ->where('attachment_id', (int) $attachment_id)
            ->delete(db_prefix() . 'deal_attachments_relations');
        return $this->db->affected_rows() > 0;
    }

    public function get_available_attachments($deal_id = null)
    {
        $this->db->select('da.*')
            ->from(db_prefix() . 'deals_attachments da');

        if ($deal_id) {
            $this->db->where('da.id NOT IN (SELECT attachment_id FROM ' . db_prefix() . 'deal_attachments_relations WHERE deal_id = ' . (int) $deal_id . ')');
        }

        return $this->db->order_by('da.dateadded', 'DESC')->get()->result_array();
    }

    // Deal Activities Methods with type
    public function get_deal_activities($deal_id)
    {
        return $this->db->where('rel_id', (int) $deal_id)
            ->order_by('date', 'DESC')
            ->get(db_prefix() . 'deals_activity')
            ->result_array();
    }

    // Dashboard helpers with visibility applied
    public function get_dashboard_deals($limit = 10)
    {
        if (staff_cant('view', 'deals')) {
            $staffid = (string) get_staff_user_id();
            $this->db->where('(' . db_prefix() . 'deals.assigned = ' . (int)$staffid
                . ' OR ' . db_prefix() . 'deals.addedfrom=' . (int)$staffid
                . ' OR ' . db_prefix() . 'deals.is_public=1' . $this->_get_visibility_where($staffid) . ')');
        }
        $this->db->select(db_prefix() . 'deals.*, dp.name as pipeline_name, ds.name as stage_name')
            ->from(db_prefix() . 'deals')
            ->join(db_prefix() . 'deals_pipelines dp', 'dp.id=' . db_prefix() . 'deals.pipeline_id', 'left')
            ->join(db_prefix() . 'deals_stages ds', 'ds.id=' . db_prefix() . 'deals.stage_id', 'left')
            ->order_by(db_prefix() . 'deals.dateadded', 'DESC')
            ->limit((int) $limit);
        return $this->db->get()->result_array();
    }

    public function get_recent_deal_activities($limit = 20)
    {
        // Filter activities by deals the user can see
        if (staff_cant('view', 'deals')) {
            $staffid = (string) get_staff_user_id();
            $this->db->where('( ' . db_prefix() . 'deals.assigned = ' . (int)$staffid
                . ' OR ' . db_prefix() . 'deals.addedfrom=' . (int)$staffid
                . ' OR ' . db_prefix() . 'deals.is_public=1' . $this->_get_visibility_where($staffid) . ' )');
        }
        $this->db->select(db_prefix() . 'deals_activity.*, s.firstname, s.lastname')
            ->from(db_prefix() . 'deals_activity')
            ->join(db_prefix() . 'deals', db_prefix() . 'deals.id=' . db_prefix() . 'deals_activity.rel_id', 'inner')
            ->join(db_prefix() . 'staff s', 's.staffid=' . db_prefix() . 'deals_activity.staffid', 'left')
            ->order_by(db_prefix() . 'deals_activity.date', 'DESC')
            ->limit((int) $limit);
        $rows = $this->db->get()->result_array();
        return array_map(function ($r) {
            $r['full_name'] = trim(($r['firstname'] ?? '') . ' ' . ($r['lastname'] ?? ''));
            return $r;
        }, $rows);
    }

    public function add_deal_activity($deal_id, $payload)
    {
        $data = [
            'title' => $payload['title'] ?? '',
            'rel_id' => (int) $deal_id,
            'description' => $payload['description'] ?? '',
            'additional_data' => $payload['additional_data'] ?? null,
            'staffid' => get_staff_user_id(),
            'full_name' => get_staff_full_name(get_staff_user_id()),
            'date' => date('Y-m-d H:i:s'),
            'type' => $payload['type'] ?? 'activity',
            'activity_date' => $payload['activity_date'] ?? null,
            'reminder' => $payload['reminder'] ?? 0,
            'owner_id' => $payload['owner_id'] ?? get_staff_user_id(),
            'guests' => $payload['guests'] ?? null,
            'status' => $payload['status'] ?? 'pending',
        ];
        $this->db->insert(db_prefix() . 'deals_activity', $data);
        return $this->db->insert_id();
    }

    // Notes
    public function add_note($deal_id, $description)
    {
        $noteData = [
            'rel_id' => (int) $deal_id,
            'description' => (string) $description,
            'addedfrom' => get_staff_user_id(),
            'dateadded' => date('Y-m-d H:i:s'),
            'date_contacted' => null,
        ];
        $this->db->insert(db_prefix() . 'deals_notes', $noteData);
        $insertId = $this->db->insert_id();
        if ($insertId) {
            $this->add_deal_activity($deal_id, [
                'description' => 'Note added',
                'additional_data' => $description,
                'type' => 'notes'
            ]);
        }
        return $insertId;
    }
    public function update_activity_status($id, $status)
    {
        $this->db->where('id', (int) $id)->update(db_prefix() . 'deals_activity', ['status' => $status]);
        return $this->db->affected_rows() > 0;
    }

    public function get_notes($deal_id)
    {
        return $this->db->where('rel_id', (int) $deal_id)
            ->order_by('dateadded', 'DESC')
            ->get(db_prefix() . 'deals_notes')
            ->result_array();
    }

    // Calls
    public function add_call($deal_id, $payload)
    {
        $data = [
            'deal_id' => (int) $deal_id,
            'outcome' => $payload['outcome'] ?? '',
            'call_notes' => $payload['call_notes'] ?? null,
            'call_date' => $payload['call_date'] ?? date('Y-m-d H:i:s'),
            'staffid' => get_staff_user_id(),
            'dateadded' => date('Y-m-d H:i:s'),
        ];
        // dd($data);
        $this->db->insert(db_prefix() . 'deals_calls', $data);
        $insertId = $this->db->insert_id();
        if ($insertId) {
            $this->add_deal_activity($deal_id, [
                'description' => 'Call logged',
                'additional_data' => json_encode([
                    'outcome' => $data['outcome'],
                    'date' => $data['call_date'],
                    'notes' => $data['call_notes']
                ]),
                'type' => 'calls'
            ]);
        }
        return $insertId;
    }

    public function get_calls($deal_id)
    {
        return $this->db->where('deal_id', (int) $deal_id)
            ->order_by('call_date', 'DESC')
            ->get(db_prefix() . 'deals_calls')
            ->result_array();
    }

    // Import support (adapted from Leads_model)
    public function import($data)
    {
        $this->db->insert(db_prefix() . 'deals', $data);
        return $this->db->insert_id();
    }

    public function get_source_id_by_name($name)
    {
        $row = $this->db->where('name', $name)->get(db_prefix() . 'deals_sources')->row();
        return $row ? (int) $row->id : null;
    }

    public function get_status_id_by_name($name)
    {
        $row = $this->db->where('name', $name)->get(db_prefix() . 'deals_statuses')->row();
        return $row ? (int) $row->id : null;
    }

    public function get_staff_id_by_name($name)
    {
        $parts = explode(' ', trim($name));
        $firstname = $parts[0] ?? '';
        $lastname = $parts[1] ?? '';
        $row = $this->db->where('firstname', $firstname)
            ->where('lastname', $lastname)
            ->get(db_prefix() . 'staff')
            ->row();
        return $row ? (int) $row->staffid : null;
    }

    public function get_sources()
    {
        return $this->db->order_by('name', 'ASC')->get(db_prefix() . 'deals_sources')->result_array();
    }

    public function get_statuses()
    {
        return $this->db->order_by('statusorder', 'ASC')->get(db_prefix() . 'deals_statuses')->result_array();
    }

    // Sources Management
    public function get_source($id = '')
    {
        if (is_numeric($id)) {
            $this->db->where('id', $id);
            return $this->db->get(db_prefix() . 'deals_sources')->row();
        }
        return $this->db->order_by('name', 'ASC')->get(db_prefix() . 'deals_sources')->result_array();
    }

    public function add_source($data)
    {
        $this->db->insert(db_prefix() . 'deals_sources', $data);
        return $this->db->insert_id();
    }

    public function update_source($data, $id)
    {
        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'deals_sources', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_source($id)
    {
        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'deals_sources');
        return $this->db->affected_rows() > 0;
    }

    // Statuses Management
    public function get_status($id = '')
    {
        if (is_numeric($id)) {
            $this->db->where('id', $id);
            return $this->db->get(db_prefix() . 'deals_statuses')->row();
        }
        $this->db->order_by('statusorder', 'asc');
        return $this->db->get(db_prefix() . 'deals_statuses')->result_array();
    }

    public function add_status($data)
    {
        $this->db->insert(db_prefix() . 'deals_statuses', $data);
        return $this->db->insert_id();
    }

    public function update_status($data, $id)
    {
        $this->db->where('id', $id);
        $this->db->update(db_prefix() . 'deals_statuses', $data);
        return $this->db->affected_rows() > 0;
    }

    public function delete_status($id)
    {
        $this->db->where('id', $id);
        $this->db->delete(db_prefix() . 'deals_statuses');
        return $this->db->affected_rows() > 0;
    }

    // Staff access check
    // public function staff_can_access_deal($id)
    // {
    //     if (staff_can('view', 'deals')) {
    //         return true;
    //     }

    //     $this->db->where('id', $id);
    //     $this->db->where('(assigned = ' . get_staff_user_id() . ' OR addedfrom=' . get_staff_user_id() . ' OR is_public=1)');
    //     $deal = $this->db->get(db_prefix() . 'deals')->row();

    //     return $deal ? true : false;
    // }

    // Get deal attachments
    // public function get_deal_attachments($deal_id)
    // {
    //     $this->db->where('deal_id', $deal_id);
    //     return $this->db->get(db_prefix() . 'deals_attachments')->result_array();
    // }

    // Get deal activities
    // public function get_deal_activities($deal_id)
    // {
    //     $this->db->where('rel_id', $deal_id);
    //     $this->db->order_by('date', 'desc');
    //     return $this->db->get(db_prefix() . 'deals_activity')->result_array();
    // }

    // Get deal notes
    public function get_deal_notes($deal_id)
    {
        $this->db->where('rel_id', $deal_id);
        $this->db->order_by('dateadded', 'desc');
        return $this->db->get(db_prefix() . 'deals_notes')->result_array();
    }

    // Get deal reminders
    public function get_deal_reminders($deal_id)
    {
        $this->db->where('deal_id', $deal_id);
        $this->db->order_by('date', 'asc');
        return $this->db->get(db_prefix() . 'deals_reminders')->result_array();
    }
}
