<?php

defined('BASEPATH') || exit('No direct script access allowed');

use app\services\AbstractKanban;

class DealsKanban extends AbstractKanban
{
    protected $pipeline_id;

    /**
     * Constructor for DealsKanban.
     * Accepts either:
     *   - ($stage_id, $pipeline_id)
     *   - (['stage_id' => ..., 'pipeline_id' => ...])
     *   - (['stage_id' => ..., 'pipeline_id' => ...], null)
     */
    public function __construct($stage_id = null, $pipeline_id = null)
    {
        // If first param is an array, extract keys
        if (is_array($stage_id)) {
            // Support both numeric and associative arrays
            if (array_key_exists('stage_id', $stage_id) || array_key_exists('pipeline_id', $stage_id)) {
                $params = $stage_id;
                $stage_id = isset($params['stage_id']) ? $params['stage_id'] : (isset($params[0]) ? $params[0] : null);
                $pipeline_id = isset($params['pipeline_id']) ? $params['pipeline_id'] : (isset($params[1]) ? $params[1] : null);
            } else {
                // Numeric array fallback
                $params = $stage_id;
                $stage_id = isset($params[0]) ? $params[0] : null;
                $pipeline_id = isset($params[1]) ? $params[1] : null;
            }
        }
        parent::__construct($stage_id);
        $this->status = $stage_id;
        $this->pipeline_id = $pipeline_id;
    }

    protected function table(): string
    {
        return 'deals';
    }

    public function defaultSortDirection()
    {
        return get_option('default_deals_kanban_sort_type') ?: 'desc';
    }

    public function defaultSortColumn()
    {
        return get_option('default_deals_kanban_sort') ?: 'dateadded';
    }

    public function limit()
    {
        return get_option('deals_kanban_limit') ?: 20;
    }

    public function where($column, $value)
    {
        $this->ci->db->where($column, $value);
        return $this;
    }

    protected function applySearchQuery($q): self
    {
        if (!startsWith($q, '#')) {
            $q = $this->ci->db->escape_like_str($q);
            $this->ci->db->where('('
                . db_prefix() . 'deals.name LIKE "%' . $q . '%" ESCAPE \'!\' OR '
                . db_prefix() . 'deals.email LIKE "%' . $q . '%" ESCAPE \'!\' OR '
                . db_prefix() . 'deals.phonenumber LIKE "%' . $q . '%" ESCAPE \'!\' OR '
                . db_prefix() . 'deals.company LIKE "%' . $q . '%" ESCAPE \'!\' OR '
                . 'CONCAT(' . db_prefix() . 'staff.firstname, \' \', ' . db_prefix() . 'staff.lastname) LIKE "%' . $q . '%" ESCAPE \'!\''
                . ')'
            );
        } else {
            $this->ci->db->where(db_prefix() . 'deals.id IN
                (SELECT rel_id FROM ' . db_prefix() . 'taggables WHERE tag_id IN
                (SELECT id FROM ' . db_prefix() . 'tags WHERE name="' . $this->ci->db->escape_str(strafter($q, '#')) . '")
                AND ' . db_prefix() . 'taggables.rel_type=\'deal\' GROUP BY rel_id HAVING COUNT(tag_id) = 1)');
        }
        return $this;
    }

    /**
     * Initiate the base query for deals in the kanban, using the correct fields and joins.
     */
    protected function initiateQuery(): self
    {
        // Select only fields that exist in the deals, staff, and deals_pipeline_relations tables as per install.php
        // Fix: Return all deals related to the selected pipeline, with their stages (do not restrict to a single stage)
        $this->ci->db->select(
            db_prefix() . 'deals.id, ' .
            db_prefix() . 'deals.name, ' .
            db_prefix() . 'deals.deal_value, ' .
            db_prefix() . 'deals.email, ' .
            db_prefix() . 'deals.phonenumber, ' .
            db_prefix() . 'deals.company, ' .
            db_prefix() . 'deals.website, ' .
            db_prefix() . 'deals.address, ' .
            db_prefix() . 'deals.city, ' .
            db_prefix() . 'deals.state, ' .
            db_prefix() . 'deals.country, ' .
            db_prefix() . 'deals.zip, ' .
            db_prefix() . 'deals.expected_close_date, ' .
            db_prefix() . 'deals.dateadded, ' .
            db_prefix() . 'deals.lastcontact, ' .
            db_prefix() . 'deals.is_imported_from_email_integration, ' .
            db_prefix() . 'deals.status_final, ' .
            db_prefix() . 'deals.business_category, ' .
            db_prefix() . 'deals.source, ' .
            db_prefix() . 'deals.assigned, ' .
            db_prefix() . 'deals.addedfrom, ' .
            db_prefix() . 'deals.is_public, ' .
            'COALESCE(' . db_prefix() . 'deals_pipeline_relations.stage_id, ' . db_prefix() . 'deals.stage_id) as rel_stage_id, ' .
            'COALESCE(' . db_prefix() . 'deals_pipeline_relations.pipeline_id, ' . db_prefix() . 'deals.pipeline_id) as rel_pipeline_id, ' .
            db_prefix() . 'deals_pipeline_relations.rating_id, ' .
            db_prefix() . 'deals_pipeline_relations.deal_id, ' .
            db_prefix() . 'deals_pipeline_relations.id as pipeline_relation_id, ' .
            db_prefix() . 'staff.staffid as assigned_staffid, ' .
            db_prefix() . 'staff.firstname as assigned_firstname, ' .
            db_prefix() . 'staff.lastname as assigned_lastname, ' .
            'CONCAT(' . db_prefix() . 'staff.firstname, " ", ' . db_prefix() . 'staff.lastname) as assigned_label, ' .
            '(SELECT GROUP_CONCAT(name SEPARATOR ",") FROM ' . db_prefix() . 'taggables JOIN ' . db_prefix() . 'tags ON ' . db_prefix() . 'taggables.tag_id = ' . db_prefix() . 'tags.id WHERE rel_id = ' . db_prefix() . 'deals.id AND rel_type="deal" ORDER BY tag_order ASC) as tags'
        );

        $this->ci->db->from(db_prefix() . 'deals');
        $this->ci->db->join(
            db_prefix() . 'deals_pipeline_relations',
            db_prefix() . 'deals_pipeline_relations.deal_id = ' . db_prefix() . 'deals.id',
            'left'
        );
        $this->ci->db->join(
            db_prefix() . 'staff',
            db_prefix() . 'staff.staffid = ' . db_prefix() . 'deals.assigned',
            'left'
        );
        // Include deals either via relation row or via main deal pipeline
        $this->ci->db->where('(
            ' . db_prefix() . 'deals_pipeline_relations.pipeline_id = ' . (int) $this->pipeline_id . ' 
            OR ' . db_prefix() . 'deals.pipeline_id = ' . (int) $this->pipeline_id . '
        )');
        // Do NOT restrict by stage here; caller groups by stage using rel_stage_id

        // Restrict by permissions: non-admin/non-view-all sees assigned/created/public OR visible via team groups/sub-groups
        if (staff_cant('view', 'deals')) {
            $staffid = (int) get_staff_user_id();
            $visibilityWhere =
                " OR EXISTS (SELECT 1 FROM " . db_prefix() . "team_groups tg " .
                "WHERE JSON_CONTAINS(" . db_prefix() . "deals.visibility_team_groups, CONCAT('\"', tg.id, '\"')) " .
                "AND (JSON_CONTAINS(tg.manager_ids, '\"$staffid\"') OR JSON_CONTAINS(tg.tl_ids, '\"$staffid\"'))) " .
                " OR EXISTS (SELECT 1 FROM " . db_prefix() . "team_sub_groups tsg " .
                "WHERE JSON_CONTAINS(" . db_prefix() . "deals.visibility_sub_groups, CONCAT('\"', tsg.id, '\"')) " .
                "AND (JSON_CONTAINS(tsg.tl_ids, '\"$staffid\"') OR JSON_CONTAINS(tsg.employee_ids, '\"$staffid\"')))";

            $this->ci->db->where('(' .
                db_prefix() . 'deals.assigned = ' . $staffid .
                ' OR ' . db_prefix() . 'deals.addedfrom = ' . $staffid .
                ' OR ' . db_prefix() . 'deals.is_public = 1' .
                $visibilityWhere .
            ')');
        }
        // Ensure unique rows; group by deal and relation id to avoid duplicates
        $this->ci->db->group_by(db_prefix() . 'deals.id, ' . db_prefix() . 'deals_pipeline_relations.id');
        return $this;
    }
}