<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Install
{
    private $CI;

    public function __construct()
    {
        $this->CI = &get_instance();
    }

    public function run()
    {
        $this->CI->load->database();

        try {

            // Create deals table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals` (
                 `id` INT NOT NULL AUTO_INCREMENT,
                    `hash` VARCHAR(65) NULL,
                    `name` VARCHAR(255) NULL,
                    `title` VARCHAR(100) NULL,
                    `company` VARCHAR(255) NULL,
                    `description` TEXT NULL,
                    `country` INT NULL,
                    `zip` VARCHAR(50) NULL,
                    `city` VARCHAR(100) NULL,
                    `state` VARCHAR(100) NULL,
                    `address` TEXT NULL,
                    `assigned` INT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `from_form_id` INT NULL,
                    `status` INT NULL,
                    `source` INT NULL,
                    `lastcontact` DATETIME NULL,
                    `dateassigned` DATE NULL,
                    `last_status_change` DATETIME NULL,
                    `addedfrom` INT NULL,
                    `email` VARCHAR(100) NULL,
                    `website` VARCHAR(255) NULL,
                    `phonenumber` VARCHAR(50) NULL,
                    `date_converted` DATETIME NULL,
                    `lost` TINYINT(1) DEFAULT 0,
                    `junk` TINYINT(1) DEFAULT 0,
                    `last_deal_status` INT NULL,
                    `is_imported_from_email_integration` TINYINT NULL,
                    `email_integration_uid` VARCHAR(30) NULL,
                    `is_public` VARCHAR(255) NULL,
                    `default_language` VARCHAR(50) NULL,
                    `client_id` INT NULL,
                    `deal_value` DECIMAL(15,2) DEFAULT 0.00,
                    `expected_close_date` DATE NULL,
                    `pipeline_id` INT NULL,
                    `stage_id` INT NULL,
                    `status_final` VARCHAR(10) NOT NULL DEFAULT "open",
                    `lost_reason_id` INT NULL,
                    `probability` TINYINT(3) DEFAULT 100,
                    `currency` INT NULL,
                    `tags` TEXT NULL,
                    `is_archived` TINYINT(1) DEFAULT 0,
                    `archived_date` DATETIME NULL,
                    `archived_by` INT NULL,
                    PRIMARY KEY (`id`),
                    KEY `idx_pipeline_id` (`pipeline_id`),
                    KEY `idx_stage_id` (`stage_id`),
                    KEY `idx_status_final` (`status_final`),
                    KEY `idx_assigned` (`assigned`),
                    KEY `idx_dateadded` (`dateadded`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            if (!$this->CI->db->table_exists(db_prefix() . 'tags_deals')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'tags_deals` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(100) NOT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            //create web to deals
            if (!$this->CI->db->table_exists(db_prefix() . 'web_to_deal')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'web_to_deal` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `name` VARCHAR(255) NOT NULL,
                `form_key` VARCHAR(32) NOT NULL UNIQUE,
                `form_data` TEXT,
                `recaptcha` TINYINT(1) DEFAULT 0,
                `language` VARCHAR(50),
                `lead_name_prefix` VARCHAR(50),
                `deal_source` INT,
                `deal_status` INT,
                `responsible` INT DEFAULT 0,
                `submit_btn_name` VARCHAR(100) DEFAULT "Submit",
                `submit_btn_bg_color` VARCHAR(7) DEFAULT "#84c529",
                `submit_btn_text_color` VARCHAR(7) DEFAULT "#ffffff",
                `submit_action` TINYINT(1) DEFAULT 0,
                `success_submit_msg` TEXT,
                `submit_redirect_url` VARCHAR(255),
                `mark_public` TINYINT(1) DEFAULT 0,
                `allow_duplicate` TINYINT(1) DEFAULT 1,
                `track_duplicate_field` VARCHAR(100),
                `track_duplicate_field_and` VARCHAR(100),
                `create_task_on_duplicate` TINYINT(1) DEFAULT 0,
                `notify_deal_imported` TINYINT(1) DEFAULT 1,
                `notify_type` ENUM("specific_staff", "roles", "assigned") DEFAULT NULL,
                `notify_ids` TEXT,
                `dateadded` DATETIME NOT NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');

                $this->CI->db->query('CREATE INDEX `idx_form_key` ON `' . db_prefix() . 'web_to_deal` (`form_key`);');
                $this->CI->db->query('CREATE INDEX `idx_deal_source` ON `' . db_prefix() . 'web_to_deal` (`deal_source`);');
                $this->CI->db->query('CREATE INDEX `idx_deal_status` ON `' . db_prefix() . 'web_to_deal` (`deal_status`);');
                $this->CI->db->query('CREATE INDEX `idx_responsible` ON `' . db_prefix() . 'web_to_deal` (`responsible`);');
            }

            // Create deals_statuses table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_statuses')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_statuses` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(100) NOT NULL,
                    `color` VARCHAR(7) NOT NULL DEFAULT "#28B8DA",
                    `statusorder` INT NOT NULL,
                    `isdefault` TINYINT(1) DEFAULT 0,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');

                // Seed default stages Open/Won/Lost
                $this->CI->db->insert(db_prefix() . 'deals_statuses', [
                    'name' => 'Open',
                    'color' => '#28B8DA',
                    'statusorder' => 1,
                    'isdefault' => 1,
                ]);
                $this->CI->db->insert(db_prefix() . 'deals_statuses', [
                    'name' => 'Won',
                    'color' => '#84c529',
                    'statusorder' => 2,
                    'isdefault' => 0,
                ]);
                $this->CI->db->insert(db_prefix() . 'deals_statuses', [
                    'name' => 'Lost',
                    'color' => '#fc2d42',
                    'statusorder' => 3,
                    'isdefault' => 0,
                ]);
            }

            // Pipelines table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_pipelines')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_pipelines` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(150) NOT NULL,
                    `order` INT DEFAULT 0,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
                // Seed default pipeline
                $this->CI->db->insert(db_prefix() . 'deals_pipelines', ['name' => 'Sales Pipeline', 'order' => 1]);
                $defaultPipelineId = $this->CI->db->insert_id();

                // Stages table linked to pipeline
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_stages` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `pipeline_id` INT NOT NULL,
                    `name` VARCHAR(150) NOT NULL,
                    `color` VARCHAR(7) NOT NULL DEFAULT "#28B8DA",
                    `position` INT DEFAULT 0,
                    `win_probability` TINYINT(3) DEFAULT 100,
                    `is_won` TINYINT(1) DEFAULT 0,
                    `is_lost` TINYINT(1) DEFAULT 0,
                    PRIMARY KEY (`id`),
                    KEY `pipeline_id` (`pipeline_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');

                // Seed default stages into default pipeline
                $this->CI->db->insert(db_prefix() . 'deals_stages', [
                    'pipeline_id' => $defaultPipelineId,
                    'name' => 'Open',
                    'color' => '#28B8DA',
                    'position' => 1,
                    'win_probability' => 100,
                    'is_won' => 0,
                    'is_lost' => 0,
                ]);
                $this->CI->db->insert(db_prefix() . 'deals_stages', [
                    'pipeline_id' => $defaultPipelineId,
                    'name' => 'Won',
                    'color' => '#84c529',
                    'position' => 2,
                    'win_probability' => 100,
                    'is_won' => 1,
                    'is_lost' => 0,
                ]);
                $this->CI->db->insert(db_prefix() . 'deals_stages', [
                    'pipeline_id' => $defaultPipelineId,
                    'name' => 'Lost',
                    'color' => '#fc2d42',
                    'position' => 3,
                    'win_probability' => 0,
                    'is_won' => 0,
                    'is_lost' => 1,
                ]);
            }

            // Create deals attachments table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_attachments')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_attachments` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `file_name` VARCHAR(255) NOT NULL,
                    `filetype` VARCHAR(100) NULL,
                    `description` TEXT NULL,
                    `staffid` INT NOT NULL,
                    `external` VARCHAR(100) NULL,
                    `external_link` TEXT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `staffid` (`staffid`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Add win_probability to deals_stages if missing
            if ($this->CI->db->table_exists(db_prefix() . 'deals_stages')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals_stages');
                if (!in_array('win_probability', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_stages` ADD `win_probability` TINYINT(3) DEFAULT 100 AFTER `position`');
                }
            }

            // Lost reasons table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_lost_reasons')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_lost_reasons` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(191) NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `name` (`name`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Add pipeline_id and stage_id to deals if missing
            if ($this->CI->db->table_exists(db_prefix() . 'deals')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals');
                if (!in_array('visibility_team_groups', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `visibility_team_groups` TEXT NULL AFTER `is_public`');
                }
                if (!in_array('visibility_sub_groups', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `visibility_sub_groups` TEXT NULL AFTER `visibility_team_groups`');
                }
                if (!in_array('pipeline_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `pipeline_id` INT NULL AFTER `client_id`');
                }
                if (!in_array('stage_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `stage_id` INT NULL AFTER `pipeline_id`');
                }
                if (!in_array('status_final', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `status_final` VARCHAR(10) NOT NULL DEFAULT "open" AFTER `stage_id`');
                }
                if (!in_array('lost_reason_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `lost_reason_id` INT NULL AFTER `status_final`');
                }
            }

            // Create deal_sources table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_sources')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_sources` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(100) NOT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }



            // Create deal_notes table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_notes')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_notes` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `rel_id` INT NOT NULL,
                    `description` TEXT NOT NULL,
                    `addedfrom` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `date_contacted` DATETIME NULL,
                    `contacted_indicator` VARCHAR(10) DEFAULT "no",
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deal_activity table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_activity')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_activity` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `rel_id` INT NOT NULL,
                    `description` VARCHAR(255) NOT NULL,
                    `additional_data` TEXT NULL,
                    `staffid` INT NULL,
                    `full_name` VARCHAR(100) NULL,
                    `date` DATETIME NOT NULL,
                    `type` VARCHAR(30) DEFAULT "activity",
                    `custom_activity` TINYINT(1) DEFAULT 0,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }
            // Ensure "type" column exists
            if ($this->CI->db->table_exists(db_prefix() . 'deals_activity')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals_activity');
                if (!in_array('type', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `type` VARCHAR(30) DEFAULT "activity" AFTER `date`');
                }
                if (!in_array('title', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `title` VARCHAR(255) NULL AFTER `id`');
                }
                if (!in_array('activity_date', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `activity_date` DATETIME NULL AFTER `title`');
                }
                if (!in_array('reminder', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `reminder` INT DEFAULT 0 AFTER `activity_date`');
                }
                if (!in_array('owner_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `owner_id` INT NULL AFTER `reminder`');
                }
                if (!in_array('guests', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `guests` TEXT NULL AFTER `owner_id`');
                }
                if (!in_array('status', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `status` VARCHAR(20) DEFAULT "pending" AFTER `guests`');
                }
            }

            // Create deal_email_integration table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_email_integration')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_email_integration` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `email` VARCHAR(100) NOT NULL,
                    `active` TINYINT(1) DEFAULT 1,
                    `created_at` DATETIME NOT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Module-specific contacts master table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_contacts_book')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_contacts_book` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `firstname` VARCHAR(100) NOT NULL,
                    `lastname` VARCHAR(100) NULL,
                    `email` VARCHAR(191) NULL,
                    `phonenumber` VARCHAR(50) NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    `active` TINYINT(1) DEFAULT 1,
                    PRIMARY KEY (`id`),
                    KEY `email` (`email`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }
            // Relation between deals and module contacts
            if (!$this->CI->db->table_exists(db_prefix() . 'deal_contact_links')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deal_contact_links` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `contact_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_contact_unique` (`deal_id`, `contact_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `contact_id` (`contact_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Module-specific companies master table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_companies_book')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_companies_book` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(255) NOT NULL,
                    `email` VARCHAR(191) NULL,
                    `phone` VARCHAR(50) NULL,
                    `website` VARCHAR(255) NULL,
                    `address` TEXT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    `active` TINYINT(1) DEFAULT 1,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `unique_email` (`email`),
                    UNIQUE KEY `unique_phone` (`phone`),
                    KEY `name` (`name`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }
            // Relation between deals and module companies
            if (!$this->CI->db->table_exists(db_prefix() . 'deal_company_links')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deal_company_links` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `company_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_company_unique` (`deal_id`, `company_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `company_id` (`company_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create companies table for storing company information
            if (!$this->CI->db->table_exists(db_prefix() . 'companies')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'companies` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(255) NOT NULL,
                    `email` VARCHAR(100) NULL,
                    `phone` VARCHAR(50) NULL,
                    `website` VARCHAR(255) NULL,
                    `address` TEXT NULL,
                    `city` VARCHAR(100) NULL,
                    `state` VARCHAR(100) NULL,
                    `zip` VARCHAR(50) NULL,
                    `country` INT NULL,
                    `description` TEXT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    `active` TINYINT(1) DEFAULT 1,
                    PRIMARY KEY (`id`),
                    KEY `name` (`name`),
                    KEY `active` (`active`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deal_attachments_relations table for linking attachments to deals
            if (!$this->CI->db->table_exists(db_prefix() . 'deal_attachments_relations')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deal_attachments_relations` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `attachment_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_attachment_unique` (`deal_id`, `attachment_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `attachment_id` (`attachment_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_forms table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_forms')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_forms` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(255) NOT NULL,
                    `form_key` VARCHAR(100) NOT NULL,
                    `lead_source` INT NULL,
                    `lead_status` INT NULL,
                    `responsible` INT NULL,
                    `language` VARCHAR(50) NULL,
                    `recaptcha` TINYINT(1) DEFAULT 0,
                    `submit_btn_name` VARCHAR(100) DEFAULT "Submit",
                    `submit_btn_bg_color` VARCHAR(7) DEFAULT "#84c529",
                    `submit_btn_text_color` VARCHAR(7) DEFAULT "#ffffff",
                    `success_submit_msg` TEXT NULL,
                    `submit_action` TINYINT(1) DEFAULT 0,
                    `submit_redirect_url` VARCHAR(255) NULL,
                    `mark_public` TINYINT(1) DEFAULT 0,
                    `allow_duplicate` TINYINT(1) DEFAULT 1,
                    `track_duplicate_field` VARCHAR(100) NULL,
                    `track_duplicate_field_and` VARCHAR(100) NULL,
                    `create_task_on_duplicate` TINYINT(1) DEFAULT 1,
                    `notify_lead_imported` TINYINT(1) DEFAULT 1,
                    `notify_type` VARCHAR(50) NULL,
                    `notify_ids` TEXT NULL,
                    `lead_name_prefix` VARCHAR(50) NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_saved_filters table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_saved_filters')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_saved_filters` (
                    `id` INT AUTO_INCREMENT PRIMARY KEY,
                    `filter_name` VARCHAR(255) NOT NULL,
                    `field` VARCHAR(100) NOT NULL,
                    `operator` VARCHAR(50) NOT NULL,
                    `value` VARCHAR(255) NOT NULL,
                    `share_with_team` TINYINT(1) DEFAULT 0,
                    `is_default` TINYINT(1) DEFAULT 0,
                    `created_by` INT NOT NULL,
                    `created_at` DATETIME NOT NULL,
                    UNIQUE KEY `filter_name` (`filter_name`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_contacts table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_contacts')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_contacts` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `contact_id` INT NOT NULL,
                    `is_primary` TINYINT(1) DEFAULT 0,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_contact` (`deal_id`, `contact_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `contact_id` (`contact_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_companies table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_companies')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_companies` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `company_id` INT NOT NULL,
                    `is_primary` TINYINT(1) DEFAULT 0,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_company` (`deal_id`, `company_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `company_id` (`company_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_products table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_products')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_products` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `product_id` INT NOT NULL,
                    `qty` DECIMAL(15,2) NOT NULL DEFAULT 1.00,
                    `rate` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                    `discount` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                    `tax` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                    `total` DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                    `description` TEXT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `product_id` (`product_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_tasks table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_tasks')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_tasks` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `task_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_task` (`deal_id`, `task_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `task_id` (`task_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_custom_fields table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_custom_fields')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_custom_fields` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `field_name` VARCHAR(100) NOT NULL,
                    `field_type` VARCHAR(50) NOT NULL,
                    `field_label` VARCHAR(255) NOT NULL,
                    `field_placeholder` VARCHAR(255) NULL,
                    `field_options` TEXT NULL,
                    `field_required` TINYINT(1) DEFAULT 0,
                    `field_order` INT DEFAULT 0,
                    `field_visible` TINYINT(1) DEFAULT 1,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_custom_field_values table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_custom_field_values')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_custom_field_values` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `field_id` INT NOT NULL,
                    `field_value` TEXT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_field` (`deal_id`, `field_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `field_id` (`field_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_reminders table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_reminders')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_reminders` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `description` TEXT NOT NULL,
                    `date` DATETIME NOT NULL,
                    `isnotified` TINYINT(1) DEFAULT 0,
                    `notify_by_email` TINYINT(1) DEFAULT 1,
                    `notify_by_sms` TINYINT(1) DEFAULT 0,
                    `creator` INT NOT NULL,
                    `datecreated` DATETIME NOT NULL,
                    `send_email` TINYINT(1) DEFAULT 0,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `date` (`date`),
                    KEY `creator` (`creator`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_sources table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_sources')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_sources` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(100) NOT NULL,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_status table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_status')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_status` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(100) NOT NULL,
                    `color` VARCHAR(7) DEFAULT "#28a745",
                    `statusorder` INT DEFAULT 1,
                    `isdefault` TINYINT(1) DEFAULT 0,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_emails table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_emails')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_emails` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `subject` VARCHAR(255) NOT NULL,
                    `message` TEXT NOT NULL,
                    `from_email` VARCHAR(100) NOT NULL,
                    `from_name` VARCHAR(100) NOT NULL,
                    `to_email` VARCHAR(100) NOT NULL,
                    `to_name` VARCHAR(100) NOT NULL,
                    `cc` TEXT NULL,
                    `bcc` TEXT NULL,
                    `attachments` TEXT NULL,
                    `sent` TINYINT(1) DEFAULT 0,
                    `sent_date` DATETIME NULL,
                    `staffid` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `staffid` (`staffid`),
                    KEY `sent` (`sent`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_calls table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_calls')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_calls` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `call_type` VARCHAR(50) NOT NULL,
                    `call_status` VARCHAR(50) NOT NULL,
                    `call_duration` INT NULL,
                    `call_recording` VARCHAR(255) NULL,
                    `call_notes` TEXT NULL,
                    `call_date` DATETIME NOT NULL,
                    `staffid` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `staffid` (`staffid`),
                    KEY `call_date` (`call_date`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_meetings table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_meetings')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_meetings` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `meeting_subject` VARCHAR(255) NOT NULL,
                    `meeting_description` TEXT NULL,
                    `meeting_date` DATETIME NOT NULL,
                    `meeting_duration` INT NULL,
                    `meeting_location` VARCHAR(255) NULL,
                    `meeting_type` VARCHAR(50) NULL,
                    `meeting_status` VARCHAR(50) NOT NULL,
                    `meeting_notes` TEXT NULL,
                    `staffid` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `staffid` (`staffid`),
                    KEY `meeting_date` (`meeting_date`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_documents table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_documents')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_documents` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `document_name` VARCHAR(255) NOT NULL,
                    `document_type` VARCHAR(100) NOT NULL,
                    `document_file` VARCHAR(255) NOT NULL,
                    `document_description` TEXT NULL,
                    `document_version` VARCHAR(20) NULL,
                    `document_status` VARCHAR(50) NOT NULL,
                    `staffid` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `staffid` (`staffid`),
                    KEY `document_status` (`document_status`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_expenses table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_expenses')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_expenses` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `expense_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_expense` (`deal_id`, `expense_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `expense_id` (`expense_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_invoices table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_invoices')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_invoices` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `invoice_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_invoice` (`deal_id`, `invoice_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `invoice_id` (`invoice_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_estimates table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_estimates')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_estimates` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `estimate_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_estimate` (`deal_id`, `estimate_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `estimate_id` (`estimate_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_proposals table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_proposals')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_proposals` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `proposal_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_proposal` (`deal_id`, `proposal_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `proposal_id` (`proposal_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_contracts table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_contracts')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_contracts` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `contract_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_contract` (`deal_id`, `contract_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `contract_id` (`contract_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_credit_notes table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_credit_notes')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_credit_notes` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `credit_note_id` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `deal_credit_note` (`deal_id`, `credit_note_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `credit_note_id` (`credit_note_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_templates table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_templates')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_templates` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `template_name` VARCHAR(255) NOT NULL,
                    `template_content` TEXT NOT NULL,
                    `template_type` VARCHAR(50) NOT NULL,
                    `template_category` VARCHAR(100) NULL,
                    `is_active` TINYINT(1) DEFAULT 1,
                    `description` TEXT NULL,
                    `variables` TEXT NULL,
                    `created_by` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `template_type` (`template_type`),
                    KEY `is_active` (`is_active`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }
            // Ensure columns exist on deals_templates for new features
            if ($this->CI->db->table_exists(db_prefix() . 'deals_templates')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals_templates');
                if (!in_array('description', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_templates` ADD `description` TEXT NULL AFTER `is_active`');
                }
                if (!in_array('variables', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_templates` ADD `variables` TEXT NULL AFTER `description`');
                }
                if (!in_array('slug', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_templates` ADD `slug` VARCHAR(255) NULL AFTER `template_name`');
                }
                if (!in_array('subject', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_templates` ADD `subject` TEXT NULL AFTER `slug`');
                }
            }

            // Create deals_automation table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_automation')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_automation` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `automation_name` VARCHAR(255) NOT NULL,
                    `automation_type` VARCHAR(50) NOT NULL,
                    `trigger_condition` TEXT NOT NULL,
                    `action_type` VARCHAR(50) NOT NULL,
                    `action_data` TEXT NOT NULL,
                    `is_active` TINYINT(1) DEFAULT 1,
                    `created_by` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `automation_type` (`automation_type`),
                    KEY `is_active` (`is_active`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_webhooks table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_webhooks')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_webhooks` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `webhook_name` VARCHAR(255) NOT NULL,
                    `webhook_url` VARCHAR(500) NOT NULL,
                    `webhook_events` TEXT NOT NULL,
                    `webhook_secret` VARCHAR(255) NULL,
                    `is_active` TINYINT(1) DEFAULT 1,
                    `created_by` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `is_active` (`is_active`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Create deals_webhook_logs table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_webhook_logs')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_webhook_logs` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `webhook_id` INT NOT NULL,
                    `event_type` VARCHAR(100) NOT NULL,
                    `payload` TEXT NOT NULL,
                    `response_code` INT NULL,
                    `response_body` TEXT NULL,
                    `status` VARCHAR(50) NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `webhook_id` (`webhook_id`),
                    KEY `event_type` (`event_type`),
                    KEY `status` (`status`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }
            // Create deals_column_order table
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_column_order')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_column_order` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `staff_id` INT NOT NULL,
                    `column_order` TEXT NOT NULL,
                    `dateupdated` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `staff_id` (`staff_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Business categories master
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_business_categories')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_business_categories` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(191) NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `name` (`name`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Ratings master (score definitions)
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_ratings')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_ratings` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(191) NOT NULL,
                    `score` INT NOT NULL DEFAULT 0,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `name` (`name`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Allow referencing category on deals
            if ($this->CI->db->table_exists(db_prefix() . 'deals')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals');
                if (!in_array('business_category', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `business_category` INT NULL AFTER `source`');
                }
            }

            // Multi-pipeline relations per deal with per-pipeline stage and score
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_pipeline_relations')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_pipeline_relations` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `pipeline_id` INT NOT NULL,
                    `stage_id` INT NULL,
                    `stage_status_id` INT NULL,
                    `stage_comment` TEXT NULL,
                    `rating_id` INT NULL,
                    `score` INT NULL,
                    `dateadded` DATETIME NOT NULL,
                    `addedfrom` INT NOT NULL,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `unique_deal_pipeline` (`deal_id`,`pipeline_id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `pipeline_id` (`pipeline_id`),
                    KEY `stage_id` (`stage_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            // Ensure new columns exist on deals_pipeline_relations
            if ($this->CI->db->table_exists(db_prefix() . 'deals_pipeline_relations')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals_pipeline_relations');
                if (!in_array('stage_status_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_pipeline_relations` ADD `stage_status_id` INT NULL AFTER `stage_id`');
                }
                if (!in_array('stage_comment', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_pipeline_relations` ADD `stage_comment` TEXT NULL AFTER `stage_status_id`');
                }
            }


            if (!$this->CI->db->table_exists(db_prefix() . 'deals_stage_statuses')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_stage_statuses` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `name` VARCHAR(255) NOT NULL,
                    `stage_id` INT NOT NULL,
                    `order` INT DEFAULT 0,
                    PRIMARY KEY (`id`),
                    INDEX `idx_stage_id` (`stage_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=' . $this->CI->db->char_set . ';');
            }

            // Create history table for stage changes
            if (!$this->CI->db->table_exists(db_prefix() . 'deals_stage_changes')) {
                $this->CI->db->query('CREATE TABLE `' . db_prefix() . 'deals_stage_changes` (
                    `id` INT NOT NULL AUTO_INCREMENT,
                    `deal_id` INT NOT NULL,
                    `pipeline_id` INT NOT NULL,
                    `from_stage_id` INT NULL,
                    `to_stage_id` INT NOT NULL,
                    `stage_status_id` INT NULL,
                    `comment` TEXT NULL,
                    `staffid` INT NOT NULL,
                    `dateadded` DATETIME NOT NULL,
                    PRIMARY KEY (`id`),
                    KEY `deal_id` (`deal_id`),
                    KEY `pipeline_id` (`pipeline_id`),
                    KEY `to_stage_id` (`to_stage_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;');
            }

            


            $fields = $this->CI->db->list_fields(db_prefix() . 'deals_activity');



            if (!in_array('title', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `title` VARCHAR(255) NULL AFTER `id`');
            }
            if (!in_array('activity_date', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `activity_date` DATETIME NULL AFTER `title`');
            }
            if (!in_array('reminder', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `reminder` INT DEFAULT 0 AFTER `activity_date`');
            }
            if (!in_array('owner_id', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `owner_id` INT NULL AFTER `reminder`');
            }
            if (!in_array('guests', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `guests` TEXT NULL AFTER `owner_id`');
            }
            if (!in_array('status', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_activity` ADD `status` VARCHAR(20) DEFAULT "pending" AFTER `guests`');
            }

            // For deals_calls, change call_status to outcome
            if (in_array('call_status', $fields = $this->CI->db->list_fields(db_prefix() . 'deals_calls'))) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_calls` CHANGE `call_status` `outcome` VARCHAR(50) DEFAULT ""');
            }

            // Add columns if not exist
            if (!in_array('call_type', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_calls` ADD `call_type` VARCHAR(50) DEFAULT NULL');
            }
            if (!in_array('call_duration', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_calls` ADD `call_duration` INT DEFAULT NULL');
            }
            if (!in_array('call_recording', $fields)) {
                $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals_calls` ADD `call_recording` VARCHAR(255) DEFAULT NULL');
            }

            if ($this->CI->db->table_exists(db_prefix() . 'deals')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals');
                if (!in_array('source', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `source` INT NULL AFTER `status_final`');
                }
                if (!in_array('status', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `status` INT NULL AFTER `source`');
                }
                if (!in_array('assigned', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `assigned` INT NULL AFTER `status`');
                }
            }

            // Add missing fields to existing tables if they don't exist

            // Update deals table
            if ($this->CI->db->table_exists(db_prefix() . 'deals')) {
                $fields = $this->CI->db->list_fields(db_prefix() . 'deals');

                if (!in_array('expected_close_date', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `expected_close_date` DATE NULL AFTER `deal_value`');
                }
                if (!in_array('pipeline_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `pipeline_id` INT NULL AFTER `expected_close_date`');
                }
                if (!in_array('stage_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `stage_id` INT NULL AFTER `pipeline_id`');
                }
                if (!in_array('status_final', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `status_final` VARCHAR(10) NOT NULL DEFAULT "open" AFTER `stage_id`');
                }
                if (!in_array('lost_reason_id', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `lost_reason_id` INT NULL AFTER `status_final`');
                }
                if (!in_array('probability', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `probability` TINYINT(3) DEFAULT 100 AFTER `lost_reason_id`');
                }
                if (!in_array('currency', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `currency` INT NULL AFTER `probability`');
                }
                if (!in_array('tags', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `tags` TEXT NULL AFTER `currency`');
                }
                if (!in_array('is_archived', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `is_archived` TINYINT(1) DEFAULT 0 AFTER `tags`');
                }
                if (!in_array('archived_date', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `archived_date` DATETIME NULL AFTER `is_archived`');
                }
                if (!in_array('archived_by', $fields)) {
                    $this->CI->db->query('ALTER TABLE `' . db_prefix() . 'deals` ADD `archived_by` INT NULL AFTER `archived_date`');
                }
            }


            // Add indexes for better query performance, only if not exist
            $indexes = [
                [db_prefix() . 'deals_sources', 'idx_name', 'name'],
                [db_prefix() . 'deals_statuses', 'idx_name', 'name'],
                [db_prefix() . 'deals_business_categories', 'idx_name', 'name'],
                [db_prefix() . 'deals_ratings', 'idx_score', 'score'],
                [db_prefix() . 'deals_stage_statuses', 'idx_name', 'name'],

            ];

            foreach ($indexes as $index) {
                list($table, $indexName, $column) = $index;
                // Check if index exists
                $query = $this->CI->db->query("SHOW INDEX FROM `$table` WHERE Key_name = '$indexName'");
                if ($query->num_rows() == 0) {
                    $this->CI->db->query("ALTER TABLE `$table` ADD INDEX `$indexName` (`$column`)");
                }
            }

            return true;
        } catch (Exception $e) {
            log_message('error', 'Deals module installation failed: ' . $e->getMessage());
            return false;
        }
    }

}

$installer = new Install();
$installer->run();

