jQuery(document).ready(function () {
    console.log('Deals Filter: Script loaded, jQuery version:', jQuery.fn.jquery);

    if (typeof Vue === 'undefined') {
        console.error('Vue.js is not loaded. Please ensure vue.min.js is included.');
        return;
    }

    // Helper: robust JSON parse (like deals_table_order.js)
    function safeParseJSON(response) {
        if (typeof response === 'object' && response !== null) return response;
        if (typeof response !== 'string') return null;
        let trimmed = response.trim();
        try {
            let parsed = JSON.parse(trimmed);
            if (parsed && parsed.final_output) {
                return JSON.parse(parsed.final_output);
            }
            return parsed;
        } catch (e) {
            // fallback: try to extract first JSON object
            let match = trimmed.match(/\{[\s\S]*\}/);
            if (match) {
                try {
                    return JSON.parse(match[0]);
                } catch (e2) {}
            }
        }
        return null;
    }

    // Vue instance for deals filter
    new Vue({
        el: '#vueApp',
        data: {
            selectedPipelines: [],
            activeFilters: [],
            pipelineStages: <?php echo json_encode(array_map(function($pipeline) {
                return [
                    'id' => $pipeline['id'],
                    'name' => $pipeline['name'],
                    'stages' => $this->deals_model->get_stages($pipeline['id']) ?? []
                ];
            }, $pipelines ?? [])) ?: '[]'; ?>,
            defaultDealsRules: <?php echo App_table::find('deals')->rulesJs() ?: '{}'; ?>,
            statuses: <?php echo json_encode($statuses ?? []) ?: '[]'; ?>,
            businessCategories: <?php echo json_encode($business_categories ?? []) ?: '[]'; ?>,
            sources: <?php echo json_encode($sources ?? []) ?: '[]'; ?>,
            ratings: <?php echo json_encode($ratings ?? []) ?: '[]'; ?>
        },
        methods: {
            setDealsPipelineFilter(pipelineId) {
                console.log('Deals Filter: Pipeline filter clicked:', pipelineId);
                const pipeline = this.pipelineStages.find(p => p && p.id == pipelineId);
                if (!pipeline) {
                    console.error('Deals Filter: Pipeline not found:', pipelineId);
                    return;
                }
                if (!this.selectedPipelines.some(p => p.id == pipelineId)) {
                    this.selectedPipelines.push(pipeline);
                    this.addFilter('pipeline_' + pipelineId, pipeline.name, 'pipeline', pipelineId);
                    console.log('Deals Filter: Added pipeline filter:', pipeline.name);
                }
                this.applyFilters();
            },
            setDealsStageFilter(pipelineId, stageId) {
                console.log('Deals Filter: Stage filter clicked:', { pipelineId, stageId });
                const pipeline = this.pipelineStages.find(p => p && p.id == pipelineId);
                if (!pipeline) {
                    console.error('Deals Filter: Pipeline not found for stage:', pipelineId);
                    return;
                }
                const stage = pipeline.stages.find(s => s && s.id == stageId);
                if (!stage) {
                    console.error('Deals Filter: Stage not found:', stageId);
                    return;
                }
                this.addFilter('stage_' + pipelineId + '_' + stageId, stage.name, 'stage', stageId, pipelineId);
                console.log('Deals Filter: Added stage filter:', stage.name);
                this.applyFilters();
            },
            setDealsStatusFilter(statusId) {
                console.log('Deals Filter: Status filter clicked:', statusId);
                const status = this.statuses.find(s => s && s.id == statusId);
                if (!status) {
                    console.error('Deals Filter: Status not found:', statusId);
                    return;
                }
                this.addFilter('status_' + statusId, status.name, 'status', statusId);
                console.log('Deals Filter: Added status filter:', status.name);
                this.applyFilters();
            },
            setDealsCategoryFilter(categoryId) {
                console.log('Deals Filter: Category filter clicked:', categoryId);
                const category = this.businessCategories.find(c => c && c.id == categoryId);
                if (!category) {
                    console.error('Deals Filter: Category not found:', categoryId);
                    return;
                }
                this.addFilter('category_' + categoryId, category.name, 'business_category', categoryId);
                console.log('Deals Filter: Added category filter:', category.name);
                this.applyFilters();
            },
            setDealsSourceFilter(sourceId) {
                console.log('Deals Filter: Source filter clicked:', sourceId);
                const source = this.sources.find(s => s && s.id == sourceId);
                if (!source) {
                    console.error('Deals Filter: Source not found:', sourceId);
                    return;
                }
                this.addFilter('source_' + sourceId, source.name, 'source', sourceId);
                console.log('Deals Filter: Added source filter:', source.name);
                this.applyFilters();
            },
            setDealsRatingFilter(ratingId) {
                console.log('Deals Filter: Rating filter clicked:', ratingId);
                const rating = this.ratings.find(r => r && r.id == ratingId);
                if (!rating) {
                    console.error('Deals Filter: Rating not found:', ratingId);
                    return;
                }
                this.addFilter('rating_' + ratingId, rating.name + ' (' + rating.score + ')', 'rating', ratingId);
                console.log('Deals Filter: Added rating filter:', rating.name);
                this.applyFilters();
            },
            addFilter(id, name, type, value, pipelineId = null) {
                if (!this.activeFilters.some(f => f.id === id)) {
                    this.activeFilters.push({ id, name, type, value, pipelineId });
                    console.log('Deals Filter: Filter added:', { id, name, type, value, pipelineId });
                }
            },
            removeFilter(filterId) {
                console.log('Deals Filter: Removing filter:', filterId);
                this.activeFilters = this.activeFilters.filter(f => f.id !== filterId);
                if (filterId.startsWith('stage_')) {
                    const pipelineId = filterId.split('_')[1];
                    if (!this.activeFilters.some(f => f.pipelineId == pipelineId && f.type === 'stage')) {
                        this.selectedPipelines = this.selectedPipelines.filter(p => p.id != pipelineId);
                        console.log('Deals Filter: Removed pipeline:', pipelineId);
                    }
                }
                this.applyFilters();
            },
            clearFilters() {
                console.log('Deals Filter: Clearing all filters');
                this.activeFilters = [];
                this.selectedPipelines = [];
                this.applyFilters();
            },
            applyFilters() {
                console.log('Deals Filter: Applying filters:', this.activeFilters);

                // Send the filters to server via window.dealsFilters and reload the server-side table
                window.dealsFilters = this.activeFilters;

                if (window.dealsTable && window.dealsTable.ajax) {
                    try {
                        window.dealsTable.ajax.reload();
                        console.log('Deals Filter: Triggered ajax.reload on dealsTable');
                    } catch (err) {
                        console.error('Deals Filter: ajax.reload failed, attempting draw()', err);
                        try { window.dealsTable.draw(false); } catch (e) {}
                    }
                } else {
                    console.warn('Deals Filter: dealsTable not initialized yet; filters will be applied on init');
                }
            }
        },
        mounted() {
            console.log('Deals Filter: Vue instance mounted');

            // Prepare server params function for initDataTable
            var serverParams = {};
            serverParams['custom_filters'] = function(data) {
                if (typeof window.dealsFilters !== 'undefined') {
                    // Send active filters as JSON string for server-side parsing
                    data.custom_filters = JSON.stringify(window.dealsFilters);
                }
            };

            // Ensure global initDataTable is available and initialize the table in server-side mode
            function tryInitDataTable() {
                if (typeof initDataTable === 'undefined') {
                    console.warn('Deals Filter: initDataTable not yet available, retrying...');
                    setTimeout(tryInitDataTable, 200);
                    return;
                }

                var notSortable = [0];
                var hiddenColumns = [];

                // Initialize and store API instance
                window.dealsTable = initDataTable('.table-deals', admin_url + 'deals/table', notSortable, hiddenColumns, serverParams, [1, 'desc']);
                console.log('Deals Filter: dealsTable initialized via initDataTable');
            }

            tryInitDataTable();

            // Initialize selectpickers for bulk actions
            if (typeof $.fn.selectpicker === 'function') {
                $('.selectpicker').selectpicker();
                console.log('Deals Filter: Bootstrap Select initialized');
            } else {
                console.error('Deals Filter: Bootstrap Select not available');
            }

            // Handle mass select all
            $('#mass_select_all').on('change', function() {
                console.log('Deals Filter: Mass select all changed:', $(this).prop('checked'));
                var checked = $(this).prop('checked');
                $('.individual').prop('checked', checked);
                toggleBulkActions();
            });

            // Handle individual checkboxes
            $('.individual').on('change', function() {
                console.log('Deals Filter: Individual checkbox changed:', $(this).val());
                toggleBulkActions();
                var allChecked = $('.individual:checked').length === $('.individual').length;
                $('#mass_select_all').prop('checked', allChecked);
            });

            function toggleBulkActions() {
                var checkedBoxes = $('.individual:checked').length;
                console.log('Deals Filter: Toggling bulk actions:', checkedBoxes);
                if (checkedBoxes > 0) {
                    $('.bulk-actions-btn').removeClass('hide');
                } else {
                    $('.bulk-actions-btn').addClass('hide');
                }
            }
        }
    });
});