/* JavaScript for Leads Table Order module */
jQuery(document).ready(function() {
    console.log('Leads Table Order: Script loaded, jQuery version:', jQuery.fn.jquery);
    console.log('Leads Table Order: DataTables version:', jQuery.fn.DataTable.version || 'Not loaded');

    if (jQuery('#leads.table-leads').length) {
        console.log('Leads Table Order: Leads table detected');
        waitForUtilities(function() {
            injectConfigureColumnsButton();
            waitForDataTable(function() {
                applyCustomColumnOrder();
            });
        }, 10000);
    } else {
        console.log('Leads Table Order: Leads table not found');
    }

    jQuery(document).on('click', '.leads-table-order-btn', function(e) {
        e.preventDefault();
        console.log('Leads Table Order: Configure Columns button clicked');
        loadColumnOrderModal();
    });
});

function waitForUtilities(callback, timeoutMs) {
    let startTime = Date.now();
    function checkUtilities() {
        if (typeof _l !== 'undefined' && typeof get_option !== 'undefined') {
            console.log('Leads Table Order: Utilities defined');
            callback();
        } else if (Date.now() - startTime > timeoutMs) {
            console.warn('Leads Table Order: Timeout waiting for utilities');
            callback();
        } else {
            setTimeout(checkUtilities, 100);
        }
    }
    checkUtilities();
}

function waitForDataTable(callback) {
    let attempts = 0;
    const maxAttempts = 200;
    function checkDataTable() {
        const $table = jQuery('#leads.table-leads');
        if ($table.length && jQuery.fn.DataTable.isDataTable($table)) {
            console.log('Leads Table Order: DataTable initialized');
            callback();
        } else if (attempts >= maxAttempts) {
            console.warn('Leads Table Order: Timeout waiting for DataTable');
            callback();
        } else {
            attempts++;
            setTimeout(checkDataTable, 100);
        }
    }
    checkDataTable();
}

function injectConfigureColumnsButton() {
    if (jQuery('.leads-table-order-btn').length) {
        console.log('Leads Table Order: Button already exists');
        return;
    }

    let $buttonsContainer = jQuery('._buttons .tw-flex.tw-justify-between');
    let $searchContainer = $buttonsContainer.find('.tw-inline');

    if (!$buttonsContainer.length || !$searchContainer.length) {
        $buttonsContainer = jQuery('._buttons');
        $searchContainer = null;
    }

    if ($buttonsContainer.length) {
        var configureText = typeof _l !== 'undefined' ? _l('leads_table_order_configure') : 'Configure Columns';
        var buttonHtml = `
            <div class="tw-flex tw-items-center tw-space-x-1 tw-mr-2">
                <a href="#" class="btn btn-default leads-table-order-btn" data-toggle="modal" data-target="#leads-table-order-modal">
                    <i class="fa fa-columns tw-mr-1"></i> ${configureText}
                </a>
            </div>
        `;
        if ($searchContainer) {
            $searchContainer.before(buttonHtml);
        } else {
            $buttonsContainer.append(buttonHtml);
        }
        console.log('Leads Table Order: Button injected');
    } else {
        console.error('Leads Table Order: Could not find _buttons container');
    }
}

function applyCustomColumnOrder() {
    console.log('Leads Table Order: Applying custom column order');
    jQuery.get(admin_url + 'leads/get_column_order')
        .done(function(response) {
            let parsedResponse;
            try {
                parsedResponse = JSON.parse(response);
            } catch (e) {
                console.error('Leads Table Order: Invalid JSON response:', response);
                alert_float('danger', 'Error parsing server response.');
                return;
            }

            if (parsedResponse.success && parsedResponse.column_order) {
                const $table = jQuery('#leads.table-leads');
                const $thead = $table.find('thead tr:last');
                const $tbody = $table.find('tbody');
                const $rows = $tbody.find('tr');

                console.log('Leads Table Order: Found', $rows.length, 'rows in tbody');

                // Map column IDs to indices
                const columnIdToIndex = {};
                const headers = jQuery('th', $thead).map(function(index) {
                    const $th = jQuery(this);
                    let colId;
                    if ($th.find('input[type="checkbox"][id="mass_select_all"]').length) {
                        colId = 'checkbox';
                    } else {
                        colId = $th.attr('id') ? $th.attr('id').replace('th-', '') : 'column_' + index;
                    }
                    columnIdToIndex[colId] = index;
                    return { id: colId, index: index };
                }).get();
                console.log('Leads Table Order: Current headers:', headers);
                console.log('Leads Table Order: Checkbox index:', columnIdToIndex['checkbox'] || 'not found');

                // Create new column order
                let columnOrder = [];
                parsedResponse.column_order.forEach(function(colId) {
                    if (columnIdToIndex[colId] !== undefined) {
                        columnOrder.push(columnIdToIndex[colId]);
                    }
                });
                if (columnIdToIndex['checkbox'] !== undefined && !parsedResponse.column_order.includes('checkbox')) {
                    columnOrder.unshift(columnIdToIndex['checkbox']);
                }
                const allColumnsCount = headers.length;
                const currentIndices = Array.from({ length: allColumnsCount }, (_, i) => i);
                const missingIndices = currentIndices.filter(i => !columnOrder.includes(i));
                columnOrder = columnOrder.concat(missingIndices);
                console.log('Leads Table Order: Server column order:', parsedResponse.column_order);
                console.log('Leads Table Order: Final column order:', columnOrder);

                // Validate row lengths
                $rows.each(function(index) {
                    const $row = jQuery(this);
                    const tdCount = $row.find('td').length;
                    if (tdCount !== allColumnsCount) {
                        console.warn(`Leads Table Order: Row ${index + 1} (ID: ${$row.attr('id')}) has ${tdCount} cells, expected ${allColumnsCount}`);
                    }
                });

                // Reorder header columns
                const $newTheadRow = jQuery('<tr role="row"></tr>');
                columnOrder.forEach(function(index) {
                    const $th = $thead.find('th').eq(index).clone(true);
                    $newTheadRow.append($th);
                });
                $table.find('thead').empty().append($newTheadRow);
                console.log('Leads Table Order: Headers reordered');

                // Reorder body columns
                const $newTbody = jQuery('<tbody></tbody>');
                $rows.each(function(index) {
                    const $row = jQuery(this);
                    const $newRow = jQuery('<tr></tr>').attr({
                        'id': $row.attr('id'),
                        'class': $row.attr('class'),
                        'role': $row.attr('role')
                    });
                    columnOrder.forEach(function(index) {
                        const $td = $row.find('td').eq(index).clone(true);
                        if ($td.length) {
                            $newRow.append($td);
                        } else {
                            console.warn(`Leads Table Order: Missing td at index ${index} in row ${index + 1}`);
                            if (headers[index].id === 'checkbox') {
                                $newRow.append('<td><div class="checkbox"><input type="checkbox" value="' + ($row.attr('id') ? $row.attr('id').replace('lead_', '') : '') + '"><label></label></div></td>');
                            } else {
                                $newRow.append('<td></td>');
                            }
                        }
                    });
                    $newTbody.append($newRow);
                });
                $tbody.replaceWith($newTbody);
                console.log('Leads Table Order: Body rows reordered, new tbody appended');

                // Rebind checkbox handlers
                $newTbody.find('td input[type="checkbox"]').each(function() {
                    const $checkbox = jQuery(this);
                    $checkbox.off('change').on('change', function() {
                        const table = $table.DataTable();
                        const row = table.row($checkbox.closest('tr'));
                        row.select($checkbox.prop('checked'));
                    });
                });

                // Redraw DataTable without destroying
                if (jQuery.fn.DataTable.isDataTable($table)) {
                    const table = $table.DataTable();
                    table.columns().every(function(index) {
                        const $th = $table.find('thead th').eq(index);
                        const colId = $th.find('input[id="mass_select_all"]').length ? 'checkbox' : ($th.attr('id') ? $th.attr('id').replace('th-', '') : 'column_' + index);
                        if (colId === 'checkbox') {
                            this.orderable(false).searchable(false);
                        }
                    });
                    if (table.colReorder) {
                        table.colReorder.order(columnOrder, true);
                    }
                    table.columns.adjust().draw(false);
                    console.log('Leads Table Order: DataTable redrawn, columns:', table.columns().indexes().toArray());
                } else {
                    console.warn('Leads Table Order: DataTable not initialized, attempting reinitialization');
                    $table.DataTable({
                        paging: true,
                        searching: true,
                        ordering: true,
                        columnDefs: [
                            {
                                targets: columnIdToIndex['checkbox'] || -1,
                                orderable: false,
                                searchable: false,
                                className: 'dt-checkboxes-cell'
                            }
                        ]
                    });
                }

                console.log('Leads Table Order: Columns and data reordered successfully');
            } else {
                console.log('Leads Table Order: No custom order or fetch failed');
            }
        })
        .fail(function(xhr) {
            console.warn('Leads Table Order: Failed to fetch column order:', xhr.responseText);
            alert_float('danger', 'Error fetching column order.');
        });
}

function loadColumnOrderModal() {
    console.log('Leads Table Order: Loading modal');
    if (!jQuery.fn.modal) {
        console.error('Leads Table Order: Bootstrap modal not loaded');
        alert('Bootstrap modal plugin missing.');
        return;
    }

    jQuery.get(admin_url + 'leads/get_column_order')
        .done(function(response) {
            let parsedResponse;
            try {
                parsedResponse = JSON.parse(response);
            } catch (e) {
                console.error('Leads Table Order: Invalid JSON response:', response);
                alert_float('danger', 'Error loading columns.');
                return;
            }

            if (parsedResponse.success) {
                var configureText = typeof _l !== 'undefined' ? _l('leads_table_order_configure') : 'Configure Columns';
                var closeText = typeof _l !== 'undefined' ? _l('close') : 'Close';
                var submitText = typeof _l !== 'undefined' ? _l('submit') : 'Save';
                var modalHtml = `
                    <div class="modal fade" id="leads-table-order-modal" tabindex="-1" role="dialog">
                        <div class="modal-dialog modal-lg">
                            <div class="modal-content">
                                <div class="modal-header">
                                    <button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">×</span></button>
                                    <h4 class="modal-title">${configureText}</h4>
                                </div>
                                <div class="modal-body">
                                    <p class="text-muted">Drag and drop to reorder columns</p>
                                    <ul id="leads-column-order-list" class="list-group sortable" style="cursor: move;">
                                        ${parsedResponse.columns.map(col => `
                                            <li class="list-group-item d-flex align-items-center" data-column-id="${col.id}">
                                                <i class="fa fa-grip-vertical mr-2"></i> ${col.name}
                                            </li>
                                        `).join('')}
                                    </ul>
                                </div>
                                <div class="modal-footer">
                                    <button type="button" class="btn btn-default" data-dismiss="modal">${closeText}</button>
                                    <button type="button" class="btn btn-primary save-column-order">
                                        <span class="btn-text">${submitText}</span>
                                        <span class="btn-loading" style="display:none;">Saving...</span>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                `;

                jQuery('#leads-table-order-modal').remove();
                jQuery('body').append(modalHtml);

                jQuery('#leads-table-order-modal').one('shown.bs.modal', function() {
                    const $list = jQuery('#leads-column-order-list');
                    if (jQuery.ui && jQuery.ui.sortable) {
                        $list.sortable({
                            items: 'li',
                            placeholder: 'list-group-item list-group-item-placeholder',
                            forcePlaceholderSize: true,
                            update: function() {
                                console.log('Leads Table Order: Columns reordered in modal');
                            }
                        });
                        console.log('Leads Table Order: Sortable initialized');
                    } else {
                        console.error('Leads Table Order: jQuery UI Sortable not loaded');
                        jQuery('.modal-body').prepend('<p class="text-danger">Drag-and-drop unavailable.</p>');
                    }
                });

                parsedResponse.column_order.forEach(function(colId) {
                    jQuery(`#leads-column-order-list li[data-column-id="${colId}"]`).appendTo('#leads-column-order-list');
                });

                jQuery('#leads-table-order-modal').modal({ backdrop: true, keyboard: true });

                jQuery('.save-column-order').on('click', function() {
                    if (!confirm('Save column order?')) return;
                    const $button = jQuery(this);
                    const $text = $button.find('.btn-text');
                    const $loading = $button.find('.btn-loading');
                    $text.hide();
                    $loading.show();
                    $button.prop('disabled', true);

                    const csrfTokenName = csrfData?.token_name || 'csrf_token_name';
                    const csrfTokenHash = csrfData?.hash || '';
                    if (!csrfTokenHash) {
                        console.error('Leads Table Order: CSRF token missing');
                        alert_float('danger', 'CSRF token unavailable.');
                        $text.show();
                        $loading.hide();
                        $button.prop('disabled', false);
                        return;
                    }

                    var columnOrder = jQuery('#leads-column-order-list li').map(function() {
                        return jQuery(this).data('column-id');
                    }).get();

                    jQuery.post(admin_url + 'leads/save_column_order', {
                        column_order: columnOrder,
                        [csrfTokenName]: csrfTokenHash
                    })
                        .done(function(response) {
                            let saveResponse;
                            try {
                                saveResponse = JSON.parse(response);
                            } catch (e) {
                                console.error('Leads Table Order: Invalid JSON response:', response);
                                alert_float('danger', 'Error saving.');
                                $text.show();
                                $loading.hide();
                                $button.prop('disabled', false);
                                return;
                            }

                            $text.show();
                            $loading.hide();
                            $button.prop('disabled', false);
                            if (saveResponse.success) {
                                alert_float('success', saveResponse.message || 'Column order saved');
                                jQuery('#leads-table-order-modal').modal('hide');
                                waitForDataTable(function() {
                                    applyCustomColumnOrder();
                                });
                            } else {
                                alert_float('danger', saveResponse.message || 'Failed to save.');
                            }
                        })
                        .fail(function(xhr) {
                            $text.show();
                            $loading.hide();
                            $button.prop('disabled', false);
                            alert_float('danger', xhr.responseText || 'Error saving.');
                            console.error('Leads Table Order: AJAX save error:', xhr.status);
                        });
                });

                jQuery('#leads-table-order-modal').on('hidden.bs.modal', function() {
                    jQuery(this).remove();
                });
            } else {
                alert_float('danger', parsedResponse.message || 'Error loading columns.');
            }
        })
        .fail(function(xhr) {
            console.error('Leads Table Order: AJAX load error:', xhr.status);
            alert_float('danger', xhr.responseText || 'Error loading columns.');
        });
}