<?php

defined('BASEPATH') || exit('No direct script access allowed');

/*
    Module Name: Deals
    Description: Deals management with list, kanban, and status (open/won/lost)
    Version: 1.0.0
    
    Requires at least: 2.9.*
*/

define('DEALS_MODULE', 'deals');

// Load installer
require_once __DIR__ . '/install.php';

$CI = &get_instance();

// Activation hook
register_activation_hook(DEALS_MODULE, 'deals_module_activation_hook');
function deals_module_activation_hook()
{
    $CI = &get_instance();
    require_once __DIR__ . '/install.php';
}

// Register language files
register_language_files(DEALS_MODULE, [DEALS_MODULE]);

// Include assets and menu
require_once __DIR__ . '/includes/assets.php';
require_once __DIR__ . '/includes/sidebar_menu_links.php';

// Register settings section and pipelines manager tab
hooks()->add_action('admin_init', function () {
    $CI = &get_instance();
    $CI->app->add_settings_section('deals', [
        'title'    => _l('deals'),
        'position' => 18,
        'children' => [
            [
                'name' => _l('pipelines'),
                'view' => 'deals/settings',
                'position' => 10,
                'icon' => 'fa-solid fa-diagram-project',
            ],
            [
                'name' => _l('lost_reasons'),
                'view' => 'deals/lost_reasons',
                'position' => 15,
                'icon' => 'fa-regular fa-face-frown',
            ],
            [
                'name' => _l('deals_sources'),
                'view' => 'deals/manage_sources',
                'position' => 20,
                'icon' => 'fa-regular fa-tag',
            ],
            [
                'name' => _l('deals_statuses'),
                'view' => 'deals/manage_statuses',
                'position' => 25,
                'icon' => 'fa-regular fa-circle-check',
            ],
            [
                'name' => _l('deals_business_categories'),
                'view' => 'deals/manage_business_categories',
                'position' => 30,
                'icon' => 'fa-regular fa-briefcase',
            ],
            [
                'name' => _l('deals_ratings'),
                'view' => 'deals/ratings',
                'position' => 35,
                'icon' => 'fa-regular fa-star',
            ],
        ],
    ]);

    // Register deals App_table (filters/rules like leads)
    if (class_exists('App_table')) {
        try {
            App_table::register(
                App_table::new('deals')->customfieldable('deals')
            );

            // Define filters/rules similar to leads
            $CI->load->model('deals/deals_model');
            $CI->load->model('staff_model');

            $rules = [
                App_table_filter::new('name', 'TextRule')->label(_l('deals_dt_name')),
                App_table_filter::new('company', 'TextRule')->label(_l('company')),
                App_table_filter::new('deal_value', 'NumberRule')->label(_l('deal_value')),
                App_table_filter::new('dateadded', 'DateRule')->label(_l('date_created')),
                App_table_filter::new('status_final', 'MultiSelectRule')->label(_l('status'))->options([
                    ['value' => 'open', 'label' => _l('open')],
                    ['value' => 'won', 'label' => _l('won')],
                    ['value' => 'lost', 'label' => _l('lost')],
                ]),
                App_table_filter::new('pipeline_id', 'SelectRule')->label(_l('pipeline'))->options(function () use ($CI) {
                    $pipelines = $CI->db->get(db_prefix() . 'deals_pipelines')->result_array();
                    return collect($pipelines)->map(fn($p) => ['value' => $p['id'], 'label' => $p['name']])->all();
                }),
                App_table_filter::new('stage_id', 'SelectRule')->label(_l('stage'))->options(function () use ($CI) {
                    $stages = $CI->db->get(db_prefix() . 'deals_stages')->result_array();
                    return collect($stages)->map(fn($s) => ['value' => $s['id'], 'label' => $s['name']])->all();
                }),
                App_table_filter::new('status', 'SelectRule')->label(_l('leads_dt_status'))->options(function () use ($CI) {
                    $statuses = $CI->deals_model->get_status();
                    return collect($statuses)->map(fn($s) => ['value' => $s['id'], 'label' => $s['name']])->all();
                }),
                App_table_filter::new('source', 'SelectRule')->label(_l('leads_source'))->options(function () use ($CI) {
                    $sources = $CI->deals_model->get_source();
                    return collect($sources)->map(fn($s) => ['value' => $s['id'], 'label' => $s['name']])->all();
                }),
            ];

            $rules[] = App_table_filter::new('assigned', 'SelectRule')->label(_l('assigned_to'))
                ->withEmptyOperators()->emptyOperatorValue(0)
                ->isVisible(fn () => staff_can('view', 'deals'))
                ->options(function () use ($CI) {
                    $staff = $CI->staff_model->get('', ['active' => 1]);
                    return collect($staff)->map(fn ($m) => ['value' => $m['staffid'], 'label' => $m['firstname'] . ' ' . $m['lastname']])->all();
                });

            App_table::find('deals')->setRules($rules)
                ->outputUsing(function ($params) {
                    extract($params);
                    $aColumns = [
                        '1',
                        db_prefix() . 'deals.id as id',
                        db_prefix() . 'deals.name as name',
                        'company',
                        db_prefix() . 'deals.deal_value as deal_value',
                        'status_final',
                        'dp.name as pipeline_name',
                        'ds.name as stage_name',
                        '(SELECT GROUP_CONCAT(name SEPARATOR ",") FROM ' . db_prefix() . 'taggables JOIN ' . db_prefix() . 'tags ON ' . db_prefix() . 'taggables.tag_id = ' . db_prefix() . 'tags.id WHERE rel_id = ' . db_prefix() . 'deals.id and rel_type="deal" ORDER by tag_order ASC LIMIT 1) as tags',
                        'dateadded',
                    ];

                    $sIndexColumn = 'id';
                    $sTable       = db_prefix() . 'deals';
                    $join = [
                        'LEFT JOIN ' . db_prefix() . 'deals_pipelines dp ON dp.id = ' . db_prefix() . 'deals.pipeline_id',
                        'LEFT JOIN ' . db_prefix() . 'deals_stages ds ON ds.id = ' . db_prefix() . 'deals.stage_id',
                    ];

                    $where = [];
                    if ($filtersWhere = $this->getWhereFromRules()) {
                        $where[] = $filtersWhere;
                    }

                    if (staff_cant('view', 'deals')) {
                        $where[] = 'AND (' . db_prefix() . 'deals.assigned = ' . get_staff_user_id() . ' OR ' . db_prefix() . 'deals.addedfrom=' . get_staff_user_id() . ' OR ' . db_prefix() . 'deals.is_public=1)';
                    }

                    $additionalColumns = [
                        db_prefix() . 'deals.addedfrom as addedfrom',
                    ];

                    $result = data_tables_init($aColumns, $sIndexColumn, $sTable, $join, $where, $additionalColumns);
                    $output  = $result['output'];
                    $rResult = $result['rResult'];

                    foreach ($rResult as $aRow) {
                        $row = [];
                        $row[] = '<div class="checkbox"><input type="checkbox" value="' . $aRow['id'] . '"><label></label></div>';
                        $hrefAttr = 'href="' . admin_url('deals/view/' . $aRow['id']) . '"';
                        $row[] = '<a ' . $hrefAttr . ' class="tw-font-medium">' . $aRow['id'] . '</a>';
                        $nameRow = '<a ' . $hrefAttr . ' class="tw-font-medium">' . e($aRow['name']) . '</a>';
                        $nameRow .= '<div class="row-options">';
                        $nameRow .= '<a ' . $hrefAttr . '>' . _l('view') . '</a>';
                        $nameRow .= ' | <a href="' . admin_url('deals/edit/' . $aRow['id']) . '">' . _l('edit') . '</a>';
                        $nameRow .= ' | <a href="' . admin_url('deals/delete/' . $aRow['id']) . '" class="_delete text-danger">' . _l('delete') . '</a>';
                        $nameRow .= '</div>';
                        $row[] = $nameRow;
                        $row[] = e($aRow['company']);
                        $row[] = '<span class="tw-font-medium">' . app_format_money((float)$aRow['deal_value'], get_base_currency()->id) . '</span>';
                        $row[] = ucfirst($aRow['status_final']);
                        $row[] = e($aRow['pipeline_name']);
                        $row[] = e($aRow['stage_name']);
                        $row[] = render_tags($aRow['tags']);
                        $row[] = '<span data-toggle="tooltip" data-title="' . e(_dt($aRow['dateadded'])) . '" class="text-has-action is-date">' . e(time_ago($aRow['dateadded'])) . '</span>';
                        $row['DT_RowId'] = 'deal_' . $aRow['id'];
                        $output['aaData'][] = $row;
                    }

                    return $output;
                });
        } catch (Exception $e) {
            // Already registered or not available; ignore
        }
    }
});

// Hooks to add 'deals' as rel_type in proposals, tasks, etc.
hooks()->add_filter('proposals_relation_types', 'add_deals_rel_type');
hooks()->add_filter('tasks_rel_types', 'add_deals_rel_type');
hooks()->add_filter('attachments_rel_types', 'add_deals_rel_type');
hooks()->add_filter('reminders_rel_types', 'add_deals_rel_type');
hooks()->add_filter('notes_rel_types', 'add_deals_rel_type');
hooks()->add_filter('activity_log_rel_types', 'add_deals_rel_type');
hooks()->add_filter('emails_rel_types', 'add_deals_rel_type');
hooks()->add_filter('calls_rel_types', 'add_deals_rel_type');

function add_deals_rel_type($types)
{
    $types[] = ['key' => 'deal', 'name' => _l('deal')];
    return $types;
}

